(* x-util.sml *)

(* Copyright (C) 2001 Alley Stoughton

   This file is part of Version 0 of an SML/NJ library for the
   pretty-printing of possibly infinite syntax trees.  See the file
   COPYING for copying and usage restrictions. *)

structure XUtil :> X_UTIL =
struct

structure EXB = EXeneBase
structure EXW = EXeneWin
structure G   = Geometry
structure W   = Widget
structure D   = Drawing
structure F   = Font
structure I   = Interact
structure FT  = FlexText

fun openDisplay dpyNameOpt =
      W.mkRoot(GetDpy.getDpy dpyNameOpt)
        handle _ => raise Fail "unable to open display"

datatype f_w_font_inf =
           FWFontInf of
             {font : EXB.font,
              wid  : int,
              ht   : int,
              asc  : int,
              des  : int}

fun fWFontInf font =
      let val wid        = F.textWidth font "a"
          val (asc, des) =
                let val {ascent, descent} = F.fontHt font
                in (ascent, descent) end
          val ht         = asc + des
      in if F.textWidth font "A" <> wid
         then raise Fail "font is not fixed width"
         else FWFontInf{font = font, wid = wid, ht = ht, asc = asc, des = des}
      end

fun coorToPt (FWFontInf{wid, ht, ...}) (FT.Coor{x, y}) =
      G.PT{x = x * wid, y = y * ht}

fun ptToCoor (FWFontInf{wid, ht, ...}) (G.PT{x, y}) =
      FT.Coor{x = x div wid, y = y div ht}

datatype draw_in = DrawIn of {pt : G.point, hghlt : bool, str : string}

fun draw(font, win, fore, back) =
      let val draw     = D.drawableOfWin win
          val normPen  =
                D.newPen[D.PV_Foreground fore, D.PV_Background back]
          val hghltPen =
                D.newPen[D.PV_Foreground back, D.PV_Background fore]
      in fn DrawIn{pt, hghlt, str} =>
              D.imageString draw
                            (if hghlt then hghltPen else normPen)
                            font
                            (pt, str)
      end

fun coorDraw(fWFI as FWFontInf{font, asc, ...}, win, fore, back) =
      let val draw = draw(font, win, fore, back)
      in fn FT.CoorDrawIn{hghlt, coor, str} =>
              let val G.PT{x, y} = coorToPt fWFI coor
              in draw(DrawIn{pt    = G.PT{x = x , y = y + asc},
                             hghlt = hghlt,
                             str   = str})
              end
      end

fun coorRectToRect (FWFontInf{wid = charWid, ht = charHt, ...})
                   (FT.CoorRect{x, y, wid, ht}) =
      G.RECT{x   = x   * charWid,
             y   = y   * charHt,
             wid = wid * charWid,
             ht  = ht  * charHt}

fun rectToCoorRect (FWFontInf{wid = charWid, ht = charHt, ...})
                   (G.RECT{x, y, wid, ht}) =
      let val x' = x + wid - 1
          val y' = y + ht - 1
          val x  = x div charWid
          val y  = y div charHt
          val x' = x' div charWid
          val y' = y' div charHt
      in FT.CoorRect{x   = x,
                     y   = y,
                     wid = x' - x + 1,
                     ht  = y' - y + 1}
      end

end;
