(* pretty.sml *)

(* Copyright (C) 2001 Alley Stoughton

   This file is part of Version 0 of an SML/NJ library for the
   pretty-printing of possibly infinite syntax trees.  See the file
   COPYING for copying and usage restrictions. *)

structure Pretty :> PRETTY =
struct

local
  open CML

  val lockCh = channel() : unit chan
  val _      = RunCML.logChannel("pretty-printing lock channel", lockCh)

  val unlockCh = channel() : unit chan
  val _        = RunCML.logChannel("pretty-printing unlock channel", unlockCh)

  fun locked() = (recv unlockCh; unlocked())

  and unlocked() = (recv lockCh; locked())
      
  val _ =
        RunCML.logServer("pretty-printing lock server",
                         fn () => ignore(spawn unlocked),
                         fn () => ())
in
  fun lock() = send(lockCh, ())

  fun unlock() = send(unlockCh, ())
end

structure FT = FlexText

datatype size = Exactly of int | AtLeast of int

datatype block = Block of data ref

and      data  = LazyData of unit -> bool * entry list
               | MeasData of size * bool * entry list

and      entry =
           Entry of
             {space  : int,
              indent : int,
              befor  : string,
              block  : block,
              after  : string}

fun block f = Block(ref(LazyData f))

fun entry (x as {space, indent, befor, block, after}) =
      Entry{space = Int.max(space, 0), indent = Int.max(indent, 0),
            befor = befor, block = block, after = after}

fun measBlock (n, Block r) =
      let val (p, y) = measData(n, !r) in r := y; p end

and measData(n, LazyData f)                 =
      let val (com, xs) = f()
      in measData(n, MeasData(AtLeast 0, com, xs)) end
  | measData(n, x as MeasData(sz, com, ys)) =
      (case sz of
            Exactly m => (Exactly m, x)
          | AtLeast m =>
              if n < m
              then (AtLeast m, x)
              else let val (sz, ys) = measEntries(true, n, ys)
                   in (sz, MeasData(sz, com, ys)) end)

and measEntries(_,   _, nil)     = (Exactly 0, nil)
  | measEntries(beg, n, x :: xs) =
      let val (sz, x) = measEntry(beg, n, x)
      in case sz of
              Exactly m =>
                if m > n
                then (AtLeast m, x :: xs)
                else let val (sz, xs) = measEntries(false, n - m, xs)
                     in case sz of
                             Exactly l => (Exactly(m + l), x :: xs)
                           | AtLeast l => (AtLeast(m + l), x :: xs)
                     end
            | AtLeast m => (AtLeast m, x :: xs)
      end

and measEntry(beg, n, x as Entry{space, indent, befor, block, after}) =
      let val m = (if beg then indent else space) + size befor + size after
      in if m > n
         then (AtLeast m, x)
         else case measBlock(n - m, block) of
                   Exactly l => (Exactly(m + l), x)
                 | AtLeast l => (AtLeast(m + l), x)
      end

fun isEmptyBlock x =
      case measBlock(0, x) of
           Exactly l => l = 0
         | AtLeast _ => false

fun blockFitsNoBreaks(wid, ind, aft, x) =
      let val avail = wid - ind - aft
      in case measBlock(avail, x) of
              Exactly l => l <= avail
            | AtLeast _ => false
      end

fun afterEntryNoBreaks(atBeg, wid, ind, aft, x) =
      let val Entry{space, indent, befor, block, after} = x
          val beforeBlock                               =
                ind +
                (if atBeg then indent else space) +
                size befor
          val avail                                     =
               wid - beforeBlock - size after - aft
      in case measBlock(avail, block) of
              Exactly l =>
                if l <= avail
                then SOME(beforeBlock + l + size after)
                else NONE
            | AtLeast _ => NONE
      end

fun abbrevSizeEntries nil                                             = NONE
  | abbrevSizeEntries (Entry{indent, befor, block, after, ...} :: xs) =
      let val n =
                indent + size befor +
                (if isEmptyBlock block then 0 else 1) +
                size after
      in case abbrevSizeEntries xs of
              NONE       => SOME(0, n)
            | SOME(m, l) => SOME(Int.max(m, n), l)
      end

fun measBlockFitsAbbrev (wid, ind, aft, Block(ref(MeasData(_, _, xs)))) =
      (case abbrevSizeEntries xs of
            NONE       => true
          | SOME(m, l) => ind + m <= wid andalso ind + l + aft <= wid)
  | measBlockFitsAbbrev _                                               =
      raise Fail "cannot happen"

fun measBlockToPartNoBreaks (prt, Block(ref(MeasData(_, _, xs)))) =
      entriesToPartNoBreaks(true, prt, xs)
  | measBlockToPartNoBreaks _                                     =
      raise Fail "cannot happen"

and entriesToPartNoBreaks(_,   prt, nil)     = prt
  | entriesToPartNoBreaks(beg, prt, x :: xs) =
      let val prt = entryToPartNoBreaks(beg, prt, x)
      in entriesToPartNoBreaks(false, prt, xs) end

and entryToPartNoBreaks(beg, prt, Entry{space, indent, befor, block, after}) =
      let val prt = FT.addSpaces(prt, if beg then indent else space)
          val prt = FT.addCharsOfStr(prt, befor)
          val prt = measBlockToPartNoBreaks(prt, block)
      in FT.addCharsOfStr(prt, after) end

datatype con =
           Con of
             {abbr  : char,
              trunc : string,
              wid   : int,
              maxHt : int}

fun abbr(Con{abbr, ...}) = abbr

fun trunc(Con{trunc, ...}) = trunc

fun wid(Con{wid, ...}) = wid

fun maxHt(Con{maxHt, ...}) = maxHt

exception Trunc of FT.part

fun checkOverflow(con, prt) =
      if FT.numRows prt = maxHt con
      then raise Trunc prt
      else prt

fun truncate(con, prt) = FT.addCharsOfStr(FT.rmCharsOrRow prt, trunc con)

fun blockToFlex(abbr, trunc, x) =
      FT.Flex(fn (wid, maxHt) =>
                   let val _   = lock()
                       val con =
                             Con{abbr = abbr, trunc = trunc, wid = wid,
                                 maxHt = maxHt}
                       val prt =
                             blockToPart(con, 0, 0, FT.empty, x)
                               handle Trunc prt => truncate(con, prt)
                       val txt = FT.complete prt
                       val _   = unlock()
                   in txt end)

and blockToPart(con, ind, aft, prt, x) =
      if blockFitsNoBreaks(wid con, ind, aft, x)
        then measBlockToPartNoBreaks(prt, x)
      else if measBlockFitsAbbrev(wid con, ind, aft, x)
        then case x of
                   Block(ref(MeasData(_, com, ys))) =>
                     if com
                     then entriesToPartSomeTopBreaksBeg(con, ind, aft, prt, ys)
                     else entriesToPartAllTopBreaks(con, ind, aft, prt, ys)
                |  _                                =>
                     raise Fail "cannot happen"
      else FT.addElision(prt, abbr con, blockToFlex(abbr con, trunc con, x))

and entriesToPartAllTopBreaks(con, ind, aft, prt, nil)     = prt
  | entriesToPartAllTopBreaks(con, ind, aft, prt, [x])     =
      entryToPartUseIndent(con, ind, aft, prt, x)
  | entriesToPartAllTopBreaks(con, ind, aft, prt, x :: xs) =
      let val prt = entryToPartUseIndent(con, ind, 0, prt, x)
          val prt = checkOverflow(con, FT.endRow prt)
          val prt = FT.addSpaces(prt, ind)
      in entriesToPartAllTopBreaks(con, ind, aft, prt, xs) end

and entriesToPartSomeTopBreaksBeg(con, ind, aft, prt, nil)     = prt
  | entriesToPartSomeTopBreaksBeg(con, ind, aft, prt, [x])     =
      entryToPartUseIndent(con, ind, aft, prt, x)
  | entriesToPartSomeTopBreaksBeg(con, ind, aft, prt, x :: xs) =
      (case afterEntryNoBreaks(true, wid con, ind, 0, x) of
            NONE   => 
              let val prt = entryToPartUseIndent(con, ind, 0, prt, x)
                  val prt = checkOverflow(con, FT.endRow prt)
                  val prt = FT.addSpaces(prt, ind)
              in entriesToPartSomeTopBreaksBeg(con, ind, aft, prt, xs) end
          | SOME l =>
              let val prt = entryToPartUseIndent(con, ind, 0, prt, x)
              in entriesToPartSomeTopBreaksMid(con, ind, l, aft, prt, xs)
              end)

and entriesToPartSomeTopBreaksMid(con, _,   _,   _,   prt, nil)     = prt
  | entriesToPartSomeTopBreaksMid(con, ind, tmp, aft, prt, x :: xs) =
      (case afterEntryNoBreaks(false, wid con, tmp,
                               if xs = nil then aft else 0, x) of
            NONE   => 
              let val prt = checkOverflow(con, FT.endRow prt)
                  val prt = FT.addSpaces(prt, ind)
              in entriesToPartSomeTopBreaksBeg(con, ind, aft, prt, x :: xs)
              end
          | SOME l =>
              let val prt = entryToPartNoBreaks(false, prt, x)
              in entriesToPartSomeTopBreaksMid(con, ind, l, aft, prt, xs)
              end)

and entryToPartUseIndent(con, ind, aft, prt, 
                         Entry{space, indent, befor, block, after}) =
      let val prt = FT.addSpaces(prt, indent)
          val prt = FT.addCharsOfStr(prt, befor)
          val prt =
                blockToPart(con, ind + indent + size befor,
                            size after + aft, prt, block)
      in FT.addCharsOfStr(prt, after) end

val empty = block(fn () => (false, nil))

fun strEntry s =
      entry{space = 0, indent = 0, befor = s, block = empty,
            after = ""}

fun entryBlock x = block(fn () => (false, [x]))

val strBlock = entryBlock o strEntry

end;
