(* flex-text.sml *)

(* Copyright (C) 2001 Alley Stoughton

   This file is part of Version 0 of an SML/NJ library for the
   pretty-printing of possibly infinite syntax trees.  See the file
   COPYING for copying and usage restrictions. *)

structure FlexText :> FLEX_TEXT =
struct

structure V  = Vector

(* function should never be called with int argument that is < 1;
   rows can't be longer than int argument *)

datatype flex = Flex of int * int -> text
and      text = Text of (char * flex option) vector vector

datatype part =
           Part of
             {numRows : int,
              rows    : (char * flex option) vector list,
              chars   : (char * flex option)list}

fun numRows(Part{numRows, ...}) = numRows

fun nullChars(Part{chars, ...}) = null chars

fun rmCharsOrRow(Part{numRows, rows, chars = nil}) =
      if numRows = 0
      then raise Fail "no rows to remove"
      else Part{numRows = numRows - 1, rows = tl rows, chars = nil}
  | rmCharsOrRow(Part{numRows, rows, chars})       =
      Part{numRows = numRows, rows = rows, chars = nil}

val empty = Part{numRows = 0, rows = nil, chars = nil}

fun endRow(Part{numRows, rows, chars}) =
      Part{numRows = numRows + 1,
           rows    = V.fromList(rev chars) :: rows,
           chars   = nil}

local
  fun addChars(s, xs) =
        let val sz = size s
  
            fun add(i, xs) =
                  if i = sz
                  then xs
                  else add(i + 1, (String.sub(s, i), NONE) :: xs)
        in add(0, xs) end
in
  fun addCharsOfStr(Part{numRows, rows, chars}, s) =
        Part{numRows = numRows, rows = rows, chars = addChars(s, chars)}
end

local
  fun addSps(0, xs) = xs
    | addSps(n, xs) = addSps(n - 1, (#" ", NONE) :: xs)
in
  fun addSpaces(Part{numRows, rows, chars}, n) =
        if n < 0
        then raise Fail "negative argument"
        else Part{numRows = numRows, rows = rows, chars = addSps(n, chars)}
end

fun addElision(Part{numRows, rows, chars}, c, x) =
      Part{numRows = numRows, rows = rows, chars = (c, SOME x) :: chars}

fun complete x =
      case endRow x of
           Part{rows, ...} => Text(V.fromList(rev rows))

datatype coor = Coor of {x : int, y : int}

fun coorLexLt(Coor{x, y}, Coor{x = x', y = y'}) =
      y < y' orelse y = y' andalso x < x'

fun coorLexGt(a, b) = coorLexLt(b, a)

fun coorLexLte(a, b) = a = b orelse coorLexLt(a, b)

fun coorLexGte(a, b) = coorLexGt(b, a)

fun width(Text rs) = V.foldl (fn (r, n) => Int.max(V.length r, n)) 0 rs

fun height(Text rs) = V.length rs

fun sub(Text rs, Coor{x, y}) =
      let val r = V.sub(rs, y)
      in V.sub(r, x) end
        handle _ => (#" ", NONE)

fun onEllipsis(text, coor) =
      case sub(text, coor) of
           (_, NONE)   => false
         | (_, SOME _) => true

datatype coor_draw_in =
           CoorDrawIn of
             {hghlt : bool,
              coor  : coor,
              str   : string}

datatype sel =
           Sel of
             {low : coor,
              hgh : coor}

fun mkSel(a, b) =
      if coorLexLte(a, b)
      then Sel{low = a, hgh = b}
      else Sel{low = b, hgh = a}

fun extractFromRow(pad, r, i, j) =
      let fun blanks n = StringCvt.padRight #" " n ""

          fun extr(i, s, s') =
                if i = j
                then if pad then s ^ s' else s
                else case SOME(V.sub(r, i)) handle _ => NONE of
                          NONE          =>
                            s ^ (if pad then s' ^ blanks(j - i) else "")
                        | SOME(#" ", _) => extr(i + 1, s, s' ^ " ")
                        | SOME(c,    _) => extr(i + 1, s ^ s' ^ str c, "")
      in extr(i, "", "") end

fun extractSel(Text rs,
               Sel{low = Coor{x, y}, hgh = Coor{x = x', y = y'}}) =
      let fun extractFromRows k =
                case SOME(V.sub(rs, k)) handle _ => NONE of
                     NONE   => nil
                   | SOME r =>
                       if k = y'
                       then [extractFromRow(false, r, 0, x')]
                       else extractFromRow(false, r, 0, (V.length r)) ^ "\n" ::
                            extractFromRows(k + 1)
      in case SOME(V.sub(rs, y)) handle _ => NONE of
              NONE   => ""
            | SOME r =>
                concat(if y = y'
                       then [extractFromRow(false, r, x, x')]
                       else (extractFromRow(false, r, Int.min(x, V.length r),
                                            (V.length r)) ^
                             "\n") ::
                            extractFromRows(y + 1))
      end

fun instrsOfSel(hghlt, lineWid, Text rs,
                Sel{low = Coor{x, y}, hgh = Coor{x = x', y = y'}}) =
      let val x  = Int.min(x,  lineWid)
          val x' = Int.min(x', lineWid)

          fun instrsOfRows k =
                let val r =
                          case SOME(V.sub(rs, k)) handle _ => NONE of
                               NONE   => #[]
                             | SOME r => r
                in if k = y'
                   then [CoorDrawIn{coor  = Coor{x = 0, y = y'},
                                    hghlt = hghlt,
                                    str   = extractFromRow(true, r, 0, x')}]
                   else CoorDrawIn{coor  = Coor{x = 0, y = k},
                                   hghlt = hghlt,
                                   str   = extractFromRow(true, r, 0,
                                                          lineWid)} ::
                        instrsOfRows(k + 1)                     
                end

          val r =
                case SOME(V.sub(rs, y)) handle _ => NONE of
                     NONE   => #[]
                   | SOME r => r
      in if y = y'
         then [CoorDrawIn{coor  = Coor{x = x, y = y},
                          hghlt = hghlt,
                          str   = extractFromRow(true, r, x, x')}]
         else CoorDrawIn{coor  = Coor{x = x, y = y},
                         hghlt = hghlt,
                         str   = extractFromRow(true, r, x, lineWid)} ::
              instrsOfRows(y + 1)
      end

fun instrsToAdjustSel(lineWid, text, old as Sel{low, hgh},
                      new as Sel{low = low', hgh = hgh'}) =
      if coorLexLte(hgh', low) orelse coorLexLte(hgh, low')
      then (instrsOfSel(false, lineWid, text, old) @
            instrsOfSel(true,  lineWid, text, new))
      else if coorLexLte(low, low')
           then if coorLexLte(hgh, hgh')
                then (* low  <= low' < hgh <= hgh' *)
                     instrsOfSel(false, lineWid, text,
                                 Sel{low = low, hgh = low'}) @
                     instrsOfSel(true, lineWid, text,
                                 Sel{low = hgh, hgh = hgh'})
                else (* low <= low' <= hgh' < hgh *)
                     instrsOfSel(false, lineWid, text,
                                 Sel{low = low, hgh = low'}) @
                     instrsOfSel(false, lineWid, text,
                                 Sel{low = hgh', hgh = hgh})
           else if coorLexLte(hgh, hgh')
                then (* low' < low <= hgh <= hgh' *)
                     instrsOfSel(true, lineWid, text,
                                 Sel{low = low', hgh = low}) @
                     instrsOfSel(true, lineWid, text,
                                 Sel{low = hgh, hgh = hgh'})
                else (* low' < low < hgh' < hgh *)
                     instrsOfSel(false, lineWid, text,
                                 Sel{low = hgh', hgh = hgh}) @
                     instrsOfSel(true, lineWid, text,
                                 Sel{low = low', hgh = low})

datatype coor_rect =
           CoorRect of
             {ht  : int,
              wid : int,
              x   : int,
              y   : int}

fun instrsOfCoorRect(Text rs, Sel{low, hgh}, CoorRect{x, y, wid, ht}) =
      let val x' = x + wid
          val y' = y + ht

          datatype relat = Before
                         | Within of int
                         | After
             
          fun relat(Coor{x, y}, k, i, j) = 
                if y < k
                  then Before
                else if y = k
                  then if x < i
                         then Before
                       else if x >= j
                         then After
                       else Within x
                else After

          fun instrsOfRow(k, r) =
                case (relat(low, k, x, x'), relat(hgh, k, x, x')) of
                     ((_,     Before) |
                      (After, _))         =>
                       [CoorDrawIn{coor  = Coor{x = x, y = k},
                                   hghlt = false,
                                   str   = extractFromRow(true, r, x, x')}]
                   | (Before, After)      =>
                       [CoorDrawIn{coor  = Coor{x = x, y = k},
                                   hghlt = true,
                                   str   = extractFromRow(true, r, x, x')}]
                   | (Before, Within j)   =>
                       [CoorDrawIn{coor  = Coor{x = x, y = k},
                                   hghlt = true,
                                   str   = extractFromRow(true, r, x, j)},
                        CoorDrawIn{coor  = Coor{x = j, y = k},
                                   hghlt = false,
                                   str   = extractFromRow(true, r, j, x')}]
                   | (Within i, After)    =>
                       [CoorDrawIn{coor  = Coor{x = x, y = k},
                                   hghlt = false,
                                   str   = extractFromRow(true, r, x, i)},
                        CoorDrawIn{coor  = Coor{x = i, y = k},
                                   hghlt = true,
                                   str   = extractFromRow(true, r, i, x')}]
                   | (Within i, Within j) =>
                       [CoorDrawIn{coor  = Coor{x = x, y = k},
                                   hghlt = false,
                                   str   = extractFromRow(true, r, x, i)},
                        CoorDrawIn{coor  = Coor{x = i, y = k},
                                   hghlt = true,
                                   str   = extractFromRow(true, r, i, j)},
                        CoorDrawIn{coor  = Coor{x = j, y = k},
                                   hghlt = false,
                                   str   = extractFromRow(true, r, j, x')}]

          fun instrsOfRows k =
                if k = y'
                then nil
                else let val r =
                               case SOME(V.sub(rs, k)) handle _ => NONE of
                                    NONE   => #[]
                                  | SOME r => r
                     in instrsOfRow(k, r) end @
                     instrsOfRows(k + 1)
      in instrsOfRows y end

end;
