(* flex-text-widget.sml *)

(* Copyright (C) 2001 Alley Stoughton

   This file is part of Version 0 of an SML/NJ library for the
   pretty-printing of possibly infinite syntax trees.  See the file
   COPYING for copying and usage restrictions. *)

structure FlexTextWidget :> FLEX_TEXT_WIDGET =
struct

open CML

structure EXB = EXeneBase
structure EXW = EXeneWin
structure G   = Geometry
structure W   = Widget
structure D   = Drawing
structure F   = Font
structure I   = Interact
structure A   = Attrs
structure FT  = FlexText
structure XU  = XUtil
structure SB  = Scrollbar
structure Q   = Quark

val attr_background     = A.attr_background
val attr_foreground     = A.attr_foreground
val attr_font           = A.attr_font
val attr_width          = A.attr_width
val attr_height         = A.attr_height
val attr_scrollbarWidth = Q.quark "scrollbarWidth"
val attr_scrollbarBack  = Q.quark "scrollbarBack"
val attr_color          = A.attr_color (* of scrollbar *)
val attr_pad            = Q.quark "pad"
val attr_divWid         = Q.quark "divWid"
val attr_cursor         = A.attr_cursor
val attr_ellipsis       = Q.quark "ellipsis"
val attr_ellipsisPress  = Q.quark "ellipsisPress"
val attr_title          = A.attr_title
val attr_iconName       = A.attr_iconName
val attr_label          = A.attr_label

val defaultBackground     = "white"
val defaultForeground     = "black"
val defaultFont           = "9x15"
val defaultWidth          = 25
val defaultHeight         = 8
val defaultScrollbarWidth = 15
val maxScrollbarWidth     = 100
val defaultScrollbarBack  = "grey"
val defaultColor          = "grey"
val defaultPad            = 2
val maxPad                = 25
val defaultDivWid         = 2
val maxDivWid             = 25
val defaultCursor         = "left_tee"
val defaultEllipsis       = "left_ptr"
val defaultEllipsisPress  = "right_ptr"
val defaultTitle          = "text"
val defaultIconName       = "text"
val defaultLabel          = "Quit"

val attrs =
      [(attr_background,     A.AT_Color,  A.AV_Str defaultBackground),
       (attr_foreground,     A.AT_Color,  A.AV_Str defaultForeground),
       (attr_font,           A.AT_Font,   A.AV_Str defaultFont),
       (attr_width,          A.AT_Int,    A.AV_Int defaultWidth),
       (attr_height,         A.AT_Int,    A.AV_Int defaultHeight),
       (attr_scrollbarWidth, A.AT_Int,    A.AV_Int defaultScrollbarWidth),
       (attr_scrollbarBack,  A.AT_Color,  A.AV_Str defaultScrollbarBack),
       (attr_color,          A.AT_Color,  A.AV_Str defaultColor),
       (attr_pad,            A.AT_Int,    A.AV_Int defaultPad),
       (attr_divWid,         A.AT_Int,    A.AV_Int defaultDivWid),
       (attr_cursor,         A.AT_Cursor, A.AV_Str defaultCursor),
       (attr_ellipsis,       A.AT_Cursor, A.AV_Str defaultEllipsis),
       (attr_ellipsisPress,  A.AT_Cursor, A.AV_Str defaultEllipsisPress),
       (attr_title,          A.AT_Str,    A.AV_Str defaultTitle),
       (attr_iconName,       A.AT_Str,    A.AV_Str defaultIconName),
       (attr_label,          A.AT_Str,    A.AV_Str defaultLabel)]

val minWidth  = 2
val minHeight = 2

datatype cursor_typ = Cursor | Ellipsis | EllipsisPress

fun setCursor(win, cursor, ellipsis, ellipsisPress) =
      let val curTypRef = ref Cursor
      in EXW.setCursor win (SOME cursor);
         fn typ =>
              if !curTypRef = typ
              then ()
              else (EXW.setCursor win
                                  (SOME(case typ of
                                             Cursor        => cursor
                                           | Ellipsis      => ellipsis
                                           | EllipsisPress => ellipsisPress));
                    curTypRef := typ)
      end

datatype cmd =
           SetFlexCmd of FT.flex
         | RealizeCmd of
             {env    : I.in_env,
              win    : EXW.window,
              sz     : G.size,
              doneCh : bool chan}

datatype init =
           Init of
             {cmdEvt        : cmd event,
              scrollEvt     : SB.scroll_evt event,
              scrollSet     : {sz : real option, top : real option} -> unit,
              background    : EXB.color,
              foreground    : EXB.color,
              cursor        : EXB.cursor,
              ellipsis      : EXB.cursor,
              ellipsisPress : EXB.cursor,
              fWFI          : XU.f_w_font_inf,
              maxDim        : int,
              flex          : FT.flex,
              display       : FT.flex -> unit}

datatype dest = ChildDest | ParentDest

datatype con =
           Con of
             {cmdEvt     : cmd event,
              scrollEvt  : SB.scroll_evt event,
              mseEvt     : (dest * I.mouse_msg)event,
              cIEvt      : (dest * I.cmd_in)event,
              scrollSet  : {sz : real option, top : real option} -> unit,
              fWFI       : XU.f_w_font_inf,
              maxDim     : int,
              child      : EXW.window,
              drawChild  : FT.coor_draw_in -> unit,
              setCursor  : cursor_typ -> unit,
              display    : FT.flex -> unit}

datatype mode =
           Normal of
             (FT.sel * unit event)option
         | Press  of
             {start : FT.coor, stop : FT.coor}

datatype dat =
           Dat of
             {parentWid : int,
              parentHt  : int,
              childHt   : int,
              topLine   : int,
              flex      : FT.flex,
              text      : FT.text,
              mode      : mode}

datatype final =
           Final of
             {cmdEvt    : cmd event,
              scrollEvt : SB.scroll_evt event,
              mseEvt    : (dest * I.mouse_msg)event,
              cIEvt     : (dest * I.cmd_in)event}

fun decodeAddrMsg x =
      case I.stripMsg x of
           I.Here x    => (ParentDest, x)
         | I.ToChild _ => (ChildDest,  I.msgBodyOf x)

fun updateFlexOfInit(Init{cmdEvt, scrollEvt, scrollSet,
                          background, foreground,
                          cursor, ellipsis, ellipsisPress,
                          fWFI, maxDim, display, ...},
                     flex) =
      Init{cmdEvt = cmdEvt, scrollEvt = scrollEvt, scrollSet = scrollSet,
           background = background, foreground = foreground,
           cursor = cursor, ellipsis = ellipsis, ellipsisPress = ellipsisPress,
           fWFI = fWFI, maxDim = maxDim, flex = flex, display = display}

fun updateTopLineOfDat(Dat{parentWid, parentHt, childHt, flex, text,
                           mode, ...},
                       topLine) =
      Dat{parentWid = parentWid, parentHt = parentHt, childHt = childHt,
          topLine = topLine, flex = flex, text = text, mode = mode}

fun updateModeOfDat(Dat{parentWid, parentHt, childHt, topLine, flex, text,
                        ...},
                    mode) =
      Dat{parentWid = parentWid, parentHt = parentHt, childHt = childHt,
          topLine = topLine, flex = flex, text = text, mode = mode}

fun moveChild(Con{child, fWFI, ...}, n) =
      let val XU.FWFontInf{ht, ...} = fWFI
          val pt                    = G.PT{x = 0, y = ~(n * ht)}
      in EXW.moveWin child pt end

fun redrawRect (Con{fWFI, drawChild, ...}, Dat{text, mode, ...}) rect =
      let val emptySel =
                FT.Sel{low = FT.Coor{x = 0, y = 0},
                       hgh = FT.Coor{x = 0, y = 0}}
          val sel      =
                case mode of
                     Normal NONE          => emptySel
                   | Normal(SOME(sel, _)) => sel
                   | Press{start, stop}   => FT.mkSel(start, stop)
          val coorRect = XU.rectToCoorRect fWFI rect
          val instrs   = FT.instrsOfCoorRect(text, sel, coorRect)
      in app drawChild instrs end

fun drawAll(Con{drawChild, ...}, Dat{parentWid, childHt, text, mode, ...}) =
      let val coorRect =
                FT.CoorRect{x   = 0,
                            y   = 0,
                            wid = parentWid,
                            ht  = childHt}
          val emptySel =
                FT.Sel{low = FT.Coor{x = 0, y = 0},
                       hgh = FT.Coor{x = 0, y = 0}}
          val sel      =
                case mode of
                     Normal NONE          => emptySel
                   | Normal(SOME(sel, _)) => sel
                   | Press{start, stop}   => FT.mkSel(start, stop)
          val instrs   = FT.instrsOfCoorRect(text, sel, coorRect)
      in app drawChild instrs end

fun restrictSelCoor(Dat{topLine, parentHt, parentWid, ...}, FT.Coor{x, y}) =
      if y < topLine
        then FT.Coor{x = 0, y = topLine}
      else if y >= topLine + parentHt
        then FT.Coor{x = 0, y = topLine + parentHt}
      else FT.Coor{x = if x < 0
                         then 0
                       else if x > parentWid
                         then parentWid
                       else x,
                   y = y}

fun scaleSize (XU.FWFontInf{wid = charWid, ht = charHt, ...})
              (G.SIZE{wid, ht}) =
      (wid div charWid, ht div charHt)

fun scroll(con as Con{scrollSet, setCursor, ...},
           dat as Dat{childHt, topLine, ...},
           newTopLine,
           set) =
      let val barTop = real newTopLine / real childHt
      in if newTopLine <> topLine
         then (moveChild(con, newTopLine);
               setCursor Cursor;
               if set
               then scrollSet{top = SOME barTop, sz = NONE}
               else ();
               updateTopLineOfDat(dat, newTopLine))
         else dat
      end

fun initState(init as Init{cmdEvt, scrollEvt, scrollSet,
                           background, foreground,
                           cursor, ellipsis, ellipsisPress,
                           fWFI, maxDim, flex, display}) =
      select
      [wrap(cmdEvt,
            fn SetFlexCmd flex                   =>
                 initState(updateFlexOfInit(init, flex))
             | RealizeCmd {env, win, sz, doneCh} =>
                 let val I.InEnv{m, ci, ...}   = I.ignoreKey env
                     val mseEvt                = wrap(m, decodeAddrMsg)
                     val cIEvt                 = wrap(ci, decodeAddrMsg)
                     val (parentWid, parentHt) = scaleSize fWFI sz
                     val text                  =
                           case flex of FT.Flex f => f(parentWid, maxDim)
                     val childHt               =
                           Int.max(parentHt, FT.height text)
                     val childCoorRect         =
                           FT.CoorRect{x   = 0,
                                       y   = 0,
                                       wid = parentWid,
                                       ht  = childHt}
                     val childRect             =
                           XU.coorRectToRect fWFI childCoorRect
                     val child                 =
                           W.wrapCreate(win, childRect,
                                        {background = SOME background})
                     val drawChild             =
                           XU.coorDraw(fWFI, child, foreground, background)
                     val con                   =
                           Con{cmdEvt     = cmdEvt,
                               scrollEvt  = scrollEvt,
                               mseEvt     = mseEvt,
                               cIEvt      = cIEvt,
                               scrollSet  = scrollSet,
                               fWFI       = fWFI,
                               maxDim     = maxDim,
                               child      = child,
                               drawChild  = drawChild,
                               setCursor  =
                                 setCursor(child, cursor,
                                           ellipsis, ellipsisPress),
                               display    = display}
                     val barSz                 = real parentHt / real childHt
                     val dat                   =
                           Dat{parentWid = parentWid,
                               parentHt  = parentHt,
                               childHt   = childHt,
                               topLine   = 0,
                               flex      = flex,
                               text      = text,
                               mode      = Normal NONE}
                 in EXW.mapWin child;
                    send(doneCh, false);
                    scrollSet{sz = SOME barSz, top = SOME 0.0};
                    mainState(con, dat)
                 end),
       wrap(scrollEvt,  (* ignore early scroll events *)
            fn _ => initState init)]

and mainState(con as Con{cmdEvt, scrollEvt, mseEvt, cIEvt, scrollSet,
                         fWFI, maxDim, child, drawChild, setCursor,
                         display},
              dat as Dat{parentWid, parentHt, childHt, topLine,
                          flex, text, mode}) =
      select
      [wrap(cmdEvt,
            fn SetFlexCmd flex         =>
                 let val text          = 
                           case flex of FT.Flex f => f(parentWid, maxDim)
                     val childHt       =
                           Int.max(parentHt, FT.height text)
                     val barSz         = real parentHt / real childHt
                     val childCoorRect =
                           FT.CoorRect{x   = 0,
                                       y   = 0,
                                       wid = parentWid,
                                       ht  = childHt}
                     val childRect     = XU.coorRectToRect fWFI childCoorRect
                     val dat           =
                           Dat{parentWid = parentWid,
                               parentHt  = parentHt,
                               childHt   = childHt,
                               topLine   = 0,
                               flex      = flex,
                               text      = text,
                               mode      = Normal NONE}
                 in EXW.moveAndResizeWin child childRect;
                    setCursor Cursor;
                    drawAll(con, dat);
                    scrollSet{sz = SOME barSz, top = SOME 0.0};
                    mainState(con, dat)
                 end
             | RealizeCmd{doneCh, ...} =>
                 (send(doneCh, true); mainState(con, dat))),
       wrap(scrollEvt,
            fn SB.ScrUp per      =>
                 let val newTopLine =
                           Int.min(childHt - parentHt,
                                   topLine + Real.trunc(per * real parentHt))
                 in mainState(con, scroll(con, dat, newTopLine, true)) end
             | SB.ScrDown per    =>
                 let val newTopLine =
                           Int.max(0,
                                   topLine - Real.trunc(per * real parentHt))
                 in mainState(con, scroll(con, dat, newTopLine, true)) end
             | (SB.ScrStart per |
                SB.ScrMove  per |
                SB.ScrEnd   per) => 
                 let val newTopLine = Real.round(per * real childHt)
                 in mainState(con, scroll(con, dat, newTopLine, false)) end),
       wrap(mseEvt,
            fn (ChildDest,
                (I.MOUSE_Enter{pt, ...}  |
                 I.MOUSE_Leave{pt, ...}  |
                 I.MOUSE_Motion{pt, ...}))                 =>
                 let val coor = XU.ptToCoor fWFI pt
                 in case mode of
                         Normal _           =>
                           (if FT.onEllipsis(text, coor)
                            then setCursor Ellipsis
                            else setCursor Cursor;
                            mainState(con, dat))
                       | Press{start, stop} =>
                           let val restrCoor = restrictSelCoor(dat, coor)
                               val oldSel    = FT.mkSel(start, stop)
                               val newSel    = FT.mkSel(start, restrCoor)
                               val instrs    =
                                     FT.instrsToAdjustSel(parentWid, text,
                                                          oldSel, newSel)
                               val newMode   =
                                     Press{start = start, stop = restrCoor}
                           in app drawChild instrs;
                              if FT.onEllipsis(text, coor)
                              then setCursor EllipsisPress
                              else setCursor Cursor;
                              mainState(con, updateModeOfDat(dat, newMode))
                           end
                 end
             | (ChildDest, I.MOUSE_FirstDown{pt, ...})    =>
                 let val coor     = XU.ptToCoor fWFI pt
                     val emptySel =
                           FT.Sel{low = FT.Coor{x = 0, y = 0},
                                  hgh = FT.Coor{x = 0, y = 0}}
                     val sel      =
                           case mode of
                                Normal(SOME(sel, _)) => sel
                              | _                    => emptySel
                     val instrs   = FT.instrsOfSel(false, parentWid, text, sel)
                     val newMode  = Press{start = coor, stop = coor}
                 in if FT.onEllipsis(text, coor)
                    then setCursor EllipsisPress
                    else setCursor Cursor;
                    app drawChild instrs;
                    mainState(con, updateModeOfDat(dat, newMode))
                 end
             | (ChildDest, I.MOUSE_LastUp{pt, time, ...}) =>
                 (case mode of
                       Normal _           => mainState(con, dat)
                     | Press{start, stop} =>
                         let val coor      = XU.ptToCoor fWFI pt
                             val restrCoor = restrictSelCoor(dat, coor)
                             val oldSel    = FT.mkSel(start, stop)
                             val newSel    = FT.mkSel(start, restrCoor)
                             val instrs    =
                                   FT.instrsToAdjustSel(parentWid, text,
                                                        oldSel, newSel)
                         in app drawChild instrs;
                            if coor = start
                            then (case FT.sub(text, coor) of
                                       (_, NONE)      => ()
                                     | (_, SOME flex) =>
                                         (spawn(fn () => display flex);
                                          setCursor Ellipsis);
                                  mainState(con,
                                            updateModeOfDat(dat, Normal NONE)))
                            else let val str     = FT.extractSel(text, newSel)
                                     val relEvt  = Select.set(child, time, str)
                                     val newMode = Normal(SOME(newSel, relEvt))
                                 in mainState(con,
                                              updateModeOfDat(dat, newMode))
                                 end
                         end)
             | _                                           =>
                 mainState(con, dat)),
       wrap(cIEvt,
            fn (ChildDest,  I.CI_Redraw rs)                     =>
                 (app (redrawRect(con, dat)) rs; mainState(con, dat))
             | (ParentDest, I.CI_Resize(G.RECT{wid, ht, ...}))  =>
                 let val (wid, ht) =
                           scaleSize fWFI (G.SIZE{wid = wid, ht = ht})
                 in if wid = parentWid andalso ht = parentHt
                    then mainState(con, dat)
                    else let val text          =
                                   if wid = parentWid
                                   then text
                                   else case flex of
                                             FT.Flex f => f(wid, maxDim)
                             val parentWid     = wid
                             val parentHt      = ht
                             val childHt       =
                                   Int.max(parentHt, FT.height text)
                             val barSz         =
                                   real parentHt / real childHt
                             val childCoorRect =
                                   FT.CoorRect{x   = 0,
                                               y   = 0,
                                               wid = parentWid,
                                               ht  = childHt}
                             val childRect     =
                                   XU.coorRectToRect fWFI childCoorRect
                             val dat           =
                                   Dat{parentWid = parentWid,
                                       parentHt  = parentHt,
                                       childHt   = childHt,
                                       topLine   = 0,
                                       flex      = flex,
                                       text      = text,
                                       mode      = Normal NONE}
                         in EXW.moveAndResizeWin child childRect;
                            setCursor Cursor;
                            drawAll(con, dat);
                            scrollSet{sz = SOME barSz, top = SOME 0.0};
                            mainState(con, dat)
                         end
                 end
             | (_, I.CI_OwnDeath)                               =>
                 let val final =
                           Final{cmdEvt    = cmdEvt,
                                 scrollEvt = scrollEvt,
                                 mseEvt    = mseEvt,
                                 cIEvt     = cIEvt}
                 in finalState final end
             | _                                                =>
                 mainState(con, dat)),
       case mode of
            Normal(SOME(sel, selRelEvt)) =>
              wrap(selRelEvt,
                   fn () =>
                         (app drawChild
                              (FT.instrsOfSel(false, parentWid, text, sel));
                          mainState(con, updateModeOfDat(dat, Normal NONE))))
          | _                            => never]

and finalState(final as Final{cmdEvt, scrollEvt, mseEvt, cIEvt}) =
      select
      [wrap(cmdEvt,    fn _ => finalState final),
       wrap(scrollEvt, fn _ => finalState final),
       wrap(mseEvt,    fn _ => finalState final),
       wrap(cIEvt,     fn _ => finalState final)]

datatype flex_text_widget =
           FlexTextWidget of
             {widget : W.widget,
              cmdCh  : cmd chan}

fun widgetOf(FlexTextWidget{widget, ...}) = widget

fun setFlex(FlexTextWidget{cmdCh, ...}, flex) = send(cmdCh, SetFlexCmd flex)

fun flexTextWidget(root, view, args) =
      let val arbitView = view

          val attrs = W.findAttr(W.attrs(view, attrs, args))

          val background = A.getColor(attrs attr_background)
          val foreground = A.getColor(attrs attr_foreground)

          val fWFI                   =
                let val font = A.getFont(attrs attr_font)
                in XU.fWFontInf font
                     handle _ =>
                              XU.fWFontInf
                              (F.openFont (W.displayOf root) defaultFont)
                end
          val XU.FWFontInf{font, ...} = fWFI

          val maxDim =
                let val XU.FWFontInf{ht, ...} = fWFI
                in 32768 div ht end

          val width  =
                let val width = A.getInt(attrs attr_width)
                in if width <= 0 orelse width > maxDim
                   then defaultWidth
                   else width
                end
          val height =
                let val height = A.getInt(attrs attr_height)
                in if height <= 0 orelse height > maxDim
                   then defaultHeight
                   else height
                end

          val scrollbarWidth =
                let val width = A.getInt(attrs attr_scrollbarWidth)
                in if width <= 0 orelse width > maxScrollbarWidth
                   then defaultScrollbarWidth
                   else width
                end
          val scrollbarBack = A.getColor(attrs attr_scrollbarBack)
          val color         = A.getColor(attrs attr_color)

          val pad =
                let val pad = A.getInt(attrs attr_pad)
                in if pad < 0 orelse pad > maxPad
                   then defaultPad
                   else pad
                end

          val divWid =
                let val divWid = A.getInt(attrs attr_divWid)
                in if divWid <= 0 orelse divWid > maxDivWid
                   then defaultDivWid
                   else divWid
                end

          val cursor        = A.getCursor(attrs attr_cursor)
          val ellipsis      = A.getCursor(attrs attr_ellipsis)
          val ellipsisPress = A.getCursor(attrs attr_ellipsisPress)
          val title         = A.getString(attrs attr_title)
          val iconName      = A.getString(attrs attr_iconName)
          val label         = A.getString(attrs attr_label)

          fun display flex =
                let val fTWArgs =
                          [(attr_background,     A.AV_Color background),
                           (attr_foreground,     A.AV_Color foreground),
                           (attr_font,           A.AV_Font font),
                           (attr_width,          A.AV_Int width),
                           (attr_height,         A.AV_Int height),
                           (attr_scrollbarWidth, A.AV_Int scrollbarWidth),
                           (attr_scrollbarBack,  A.AV_Color scrollbarBack),
                           (attr_color,          A.AV_Color color),
                           (attr_pad,            A.AV_Int pad),
                           (attr_divWid,         A.AV_Int divWid),
                           (attr_cursor,         A.AV_Cursor cursor),
                           (attr_ellipsis,       A.AV_Cursor ellipsis),
                           (attr_ellipsisPress,  A.AV_Cursor ellipsisPress),
                           (attr_title,          A.AV_Str title),
                           (attr_iconName,       A.AV_Str iconName),
                           (attr_label,          A.AV_Str label)]

                    val dspArgs =
                          [(attr_title,      A.AV_Str title),
                           (attr_iconName,   A.AV_Str iconName),
                           (attr_background, A.AV_Color background),
                           (attr_foreground, A.AV_Color foreground),
                           (attr_pad,        A.AV_Int pad),
                           (attr_font,       A.AV_Font font),
                           (attr_label,      A.AV_Str label),
                           (attr_divWid,     A.AV_Int divWid)]

                    val fTW = flexTextWidget(root, arbitView, fTWArgs)
                in setFlex(fTW, flex);
                   DisplayWidget.display (root, arbitView, dspArgs)
                                         (widgetOf fTW, nil)
                end

          val scrollArgs =
                [(Attrs.attr_width,      Attrs.AV_Int scrollbarWidth),
                 (Attrs.attr_background, Attrs.AV_Color scrollbarBack),
                 (Attrs.attr_color,      Attrs.AV_Color color)]
          val scrollbar = SB.vScrollbar(root, arbitView, scrollArgs)
          val scrollEvt = SB.evtOf scrollbar
          val scrollSet = SB.setVals scrollbar

          val cmdCh = channel() : cmd chan

          val initFlex = FT.Flex(fn _ => FT.Text #[])
          val init     =
                Init{cmdEvt        = recvEvt cmdCh,
                     scrollEvt     = scrollEvt,
                     scrollSet     = scrollSet,
                     background    = background,
                     foreground    = foreground,
                     cursor        = cursor,
                     ellipsis      = ellipsis,
                     ellipsisPress = ellipsisPress,
                     fWFI          = fWFI,
                     maxDim        = maxDim,
                     flex          = initFlex,
                     display       = display}
          val _        = spawn(fn () => initState init)

          fun textArgs() = {background = SOME background}

          fun textBoundsOf() =
                let val XU.FWFontInf{wid, ht, ...} = fWFI

                    val xDim =
                          W.DIM{base = 0, incr = wid, min = minWidth,
                                nat = width, max = NONE}
                    val yDim =
                          W.DIM{base = 0, incr = ht, min = minHeight,
                                nat = height, max = NONE}
                in W.mkBounds{x_dim = xDim, y_dim = yDim} end

          fun textRealize{env, win, sz} =
                let val doneCh = channel()
                in send(cmdCh,
                        RealizeCmd{env = env, win = win, sz = sz,
                                   doneCh = doneCh});
                   if recv doneCh
                   then raise W.AlreadyRealized
                   else ()
                end

         val text =
               W.mkWidget{root     = root,
                          args     = textArgs,
                          boundsOf = textBoundsOf,
                          realize  = textRealize}

         val divArgs =
               [(A.attr_width,      A.AV_Int divWid),
                (A.attr_foreground, A.AV_Color foreground)]
         val divider =
               Divider.vertDivider(root, arbitView, divArgs)

         val glue = Box.Glue{nat = pad, min = pad, max = SOME pad}

         val layoutArgs = nil
         val layout     =
               Box.layout (root, arbitView, layoutArgs)
                          (Box.HzCenter
                           [Box.WBox(SB.widgetOf scrollbar),
                            glue,
                            Box.WBox divider,
                            glue,
                            Box.WBox text])
      in FlexTextWidget{widget = Box.widgetOf layout,
                        cmdCh  = cmdCh}
      end

end;
