(* display-widget.sml *)

(* Copyright (C) 2001, 2005 Alley Stoughton

   This file is part of Version 0 of an SML/NJ library for the
   pretty-printing of possibly infinite syntax trees.  See the file
   COPYING for copying and usage restrictions. *)

structure DisplayWidget :> DISPLAY_WIDGET =
struct

open CML

structure EXB  = EXeneBase
structure Bttn = Button
structure W    = Widget
structure A    = Attrs
structure Q    = Quark

val attr_title      = A.attr_title
val attr_iconName   = A.attr_iconName
val attr_background = A.attr_background
val attr_foreground = A.attr_foreground
val attr_pad        = Q.quark "pad"
val attr_font       = A.attr_font
val attr_label      = A.attr_label
val attr_bttnSep    = Q.quark "bttnSep"
val attr_divWid     = Q.quark "divWid"

val defaultTitle      = "displayed widget"
val defaultIconName   = "displayed widget"
val defaultBackground = "white"
val defaultForeground = "black"
val defaultPad        = 2
val maxPad            = 25
val defaultFont       = "8x13"
val defaultLabel      = "Quit"
val defaultBttnSep    = 8
val maxBttnSep        = 25
val defaultDivWid     = 2
val maxDivWid         = 25

val attrs =
      [(attr_title,      A.AT_Str,   A.AV_Str defaultTitle),
       (attr_iconName,   A.AT_Str,   A.AV_Str defaultIconName),
       (attr_background, A.AT_Color, A.AV_Str defaultBackground),
       (attr_foreground, A.AT_Color, A.AV_Str defaultForeground),
       (attr_pad,        A.AT_Int,   A.AV_Int defaultPad),
       (attr_font,       A.AT_Font,  A.AV_Str defaultFont),
       (attr_label,      A.AT_Str,   A.AV_Str defaultLabel),
       (attr_bttnSep,    A.AT_Int,   A.AV_Int defaultBttnSep),
       (attr_divWid,     A.AT_Int,   A.AV_Int defaultDivWid)]

val bttnToRigidBox = Box.WBox o Shape.mkRigid o Bttn.widgetOf

fun bttnsToRowBox(sep, bs) =
      let val sepGlue = Box.Glue{nat = sep, min = sep, max = SOME sep}
          val endGlue = Box.Glue{nat = 0,   min = 0,   max = NONE}

          fun bToRB nil       = raise Fail "cannot happen"
            | bToRB [b]       = bttnToRigidBox b :: [endGlue]
            | bToRB (b :: bs) = bttnToRigidBox b :: sepGlue :: bToRB bs
      in Box.HzCenter(bToRB bs) end

fun display (root, view, args) (widget, bttns) =
      let val arbitView = view

          val attrs      = W.findAttr(W.attrs(view, attrs, args))

          val title      = A.getString(attrs attr_title)
          val iconName   = A.getString(attrs attr_iconName)
          val background = A.getColor(attrs attr_background)
          val foreground = A.getColor(attrs attr_foreground)

          val pad =
                let val pad = A.getInt(attrs attr_pad)
                in if pad < 0 orelse pad > maxPad
                   then defaultPad
                   else pad
                end

          val font  = A.getFont(attrs attr_font)
          val label = A.getString(attrs attr_label)

          val bttnSep = 
                let val bttnSep = A.getInt(attrs attr_bttnSep)
                in if bttnSep <= 0 orelse bttnSep > maxBttnSep
                   then defaultBttnSep
                   else bttnSep
                end

          val divWid =
                let val divWid = A.getInt(attrs attr_divWid)
                in if divWid <= 0 orelse divWid > maxDivWid
                   then defaultDivWid
                   else divWid
                end

          val padGlue = Box.Glue{nat = pad, min = pad, max = SOME pad}

          val quitBttnArgs  =
                [(A.attr_background, A.AV_Color background),
                 (A.attr_foreground, A.AV_Color foreground),
                 (A.attr_font,       A.AV_Font font),
                 (A.attr_label,      A.AV_Str label)]
          val quitBttn      = Bttn.textBtn(root, arbitView, quitBttnArgs)
          val quitBttnEvt   = Bttn.evtOf quitBttn

          val divArgs =
                [(A.attr_width,      A.AV_Int divWid),
                 (A.attr_foreground, A.AV_Color foreground)]
          val divider = Divider.horzDivider(root, arbitView, divArgs)

          val bttnRow = bttnsToRowBox(bttnSep, quitBttn :: bttns)

          val box =
                Box.HzCenter
                [padGlue,
                 Box.VtLeft
                 [padGlue,
                  bttnRow,
                  padGlue,
                  Box.WBox divider,
                  padGlue,
                  Box.WBox widget,
                  padGlue],
                 padGlue]

          val layoutArgs = nil
          val layout     = Box.layout (root, arbitView, layoutArgs) box

          val shellArgs =
                [(A.attr_title,      A.AV_Str title),
                 (A.attr_iconName,   A.AV_Str iconName),
                 (A.attr_background, A.AV_Color background)]
          val shell     =
                Shell.shell (root, arbitView, shellArgs)
                            (Box.widgetOf layout)
          val delEvt    = Shell.deleteEvent shell
          val _         = Shell.init shell

          fun waitQuitOrDel() =
                select
                [wrap(quitBttnEvt,
                      fn msg =>
                           case msg of
                                Bttn.BtnUp _ => ()
                              | _            => waitQuitOrDel()),
                 wrap(delEvt, fn () => ())]
      in waitQuitOrDel();
         Shell.destroy shell
      end

end;
