(* simp-fun.sml *)

(* Copyright (C) 2001 Alley Stoughton

   This file is part of Version 0 of an SML/NJ library for the
   pretty-printing of possibly infinite syntax trees.  See the file
   COPYING for copying and usage restrictions. *)

structure SimpFun :> SIMP_FUN =
struct

structure P = Pretty

type 'a thunk = unit -> 'a

datatype exp = Id   of string
             | Num  of string
             | Abs  of string * exp thunk
             | App  of exp * exp thunk
             | Cond of exp thunk * exp thunk * exp thunk
             | List of exp thunk list

fun thunk x = fn () => x

fun abs(s, x) = Abs(s, thunk x)

fun app(x, y) = App(x, thunk y)

fun cond(x, y, z) = Cond(thunk x, thunk y, thunk z)

fun list xs = List(map thunk xs)

fun prtyExp (Abs(s, f))     =
      P.block(fn () =>
                   (false,
                    [P.entry{space = 0, indent = 0, befor = "fn ",
                             block = P.strBlock s, after = " =>"},
                     P.entry{space = 1, indent = 5, befor = "",
                             block = prtyExp(f()), after = ""}]))
  | prtyExp (Cond(f, g, h)) =
      P.block(fn () =>
                   (false,
                    [P.entry{space = 0, indent = 0, befor = "if ",
                             block = prtyExp(f()), after = ""},
                     P.entry{space = 1, indent = 0, befor = "then ",
                             block = prtyExp(g()), after = ""},
                     P.entry{space = 1, indent = 0, befor = "else ",
                             block = prtyExp(h()), after = ""}]))
  | prtyExp x               = prtyApp x

and prtyApp (x as App _)  =
      let fun left (App(y, f)) =
                     left y @
                     [P.entry{space = 1, indent = 0, befor = "",
                              block = prtyAtom(f()), after = ""}]
            | left y           =
                     [P.entry{space = 0, indent = 0, befor = "",
                              block = prtyAtom y, after = ""}]
      in P.block(fn () => (true, left x)) end
  | prtyApp x             = prtyAtom x

and prtyAtom (Id s)     = P.strBlock s
  | prtyAtom (Num s)    = P.strBlock s
  | prtyAtom (List nil) = P.strBlock "[]"
  | prtyAtom (List[f])  =
      P.entryBlock(P.entry{space = 0, indent = 0, befor = "[",
                           block = prtyExp(f()), after = "]"})
  | prtyAtom (List es)  =
      let val f  = hd es
          val gs = List.take(tl es, length es - 2)
          val h  = List.last es

          fun middle nil       = nil
            | middle (g :: gs) =
                P.entry{space = 1, indent = 1, befor = "",
                        block = prtyExp(g()), after = ","}   ::
                middle gs
      in P.block(fn () =>
                      (true,
                       [P.entry{space = 0, indent = 0, befor = "[",
                                block = prtyExp(f()), after = ","}]   @
                       middle gs                                      @
                       [P.entry{space = 1, indent = 1, befor = "",
                                block = prtyExp(h()), after = "]"}]))
      end
  | prtyAtom x          =
      P.entryBlock(P.entry{space = 0, indent = 0, befor = "(",
                           block = prtyExp x, after = ")"})

val pretty = prtyExp

end;
