(* main.sml *)

(* Copyright (C) 2001, 2005 Alley Stoughton

   This file is part of Version 0 of an SML/NJ library for the
   pretty-printing of possibly infinite syntax trees.  See the file
   COPYING for copying and usage restrictions. *)

structure Main :> MAIN =
struct

open CML

structure EXB = EXeneBase
structure FTW = FlexTextWidget
structure W   = Widget

val prettyTM : TraceCML.trace_module =
      TraceCML.traceModule(XDebug.eXeneTM, "pretty")

fun trace f = TraceCML.trace (prettyTM, f)

fun display(root, view, exp) =
      let val block = SimpFun.pretty exp
          val flex  = Pretty.blockToFlex(#"?", "<truncated>", block)
          val ftw   = FTW.flexTextWidget(root, view, [])
      in FTW.setFlex(ftw, flex);
         spawn(fn () =>
                    DisplayWidget.display (root, view, nil)
                                          (FTW.widgetOf ftw, nil));
         ()
      end

fun error(root, prog, msg) =
      (TextIO.output(TextIO.stdErr, prog ^ ": " ^ msg ^ "\n");
       W.delRoot root;
       RunCML.shutdown OS.Process.failure)

datatype read = EOF | BadInt | GoodInt of int

fun readInt() =
      case TextIO.inputLine TextIO.stdIn of
           NONE   => EOF
         | SOME s =>
             (case Int.fromString s of
                   NONE   => BadInt
                 | SOME n => GoodInt n)
               handle Overflow => BadInt

fun loop(root, view) =
      (print("enter next syntax tree (0-" ^
             Int.toString(length Examps.examps - 1) ^
             ") (or signal EOF to exit): ");
       case readInt() of
            EOF       => ()
          | BadInt    => (print "bad integer\n"; loop(root, view))
          | GoodInt n =>
              (if n < 0 orelse n > length Examps.examps - 1
               then (print "number out of range\n";
                     loop(root, view))
               else (display(root, view, List.nth(Examps.examps, n));
                     loop(root, view))))

val optSpec =  
      [(Styles.OPT_NAMED("display"), "-display", Styles.OPT_SEPARG,
       Attrs.AT_Str),
       (Styles.OPT_NAMED("debug"), "-debug", Styles.OPT_NOARG("on"),
       Attrs.AT_Bool),
       (Styles.OPT_NAMED("name"), "-name", Styles.OPT_SEPARG,
       Attrs.AT_Str),
       (Styles.OPT_RESSPEC("*background"), "-background", Styles.OPT_SEPARG,
       Attrs.AT_Str),
       (Styles.OPT_RESSPEC("*background"), "-bg", Styles.OPT_SEPARG,
       Attrs.AT_Str),
       (Styles.OPT_RESSPEC("*foreground"), "-foreground", Styles.OPT_SEPARG,
       Attrs.AT_Str),
       (Styles.OPT_RESSPEC("*foreground"), "-fg", Styles.OPT_SEPARG,
       Attrs.AT_Str),
       (Styles.OPT_RESSPEC("*borderWidth"), "-border", Styles.OPT_SEPARG,
       Attrs.AT_Str),
       (Styles.OPT_RESSPEC("*font"), "-font", Styles.OPT_SEPARG,
       Attrs.AT_Font),
       (Styles.OPT_RESSPEC("*font"), "-fn", Styles.OPT_SEPARG,
       Attrs.AT_Font)]

fun start(prog, args) =
      let val (optDB, unArgs) = Widget.parseCommand optSpec args

          val _ =
                if not(null unArgs)
                then TextIO.print(prog ^ ": some arguments ignored\n")
                else ()

          val displayNameOpt =
                case Widget.findNamedOptStrings optDB
                                                (Styles.OPT_NAMED("display"))
                of
                     []     => NONE
                   | s :: _ => SOME s

          val root =
                W.mkRoot(GetDpy.getDpy displayNameOpt)
                  handle EXB.BadAddr _ =>
                           (TextIO.output(TextIO.stdOut,
                                          prog ^ ": " ^
                                          "unable to open display: " ^
                                          (case displayNameOpt of
                                                NONE   => "[default]"
                                              | SOME s => s) ^
                                          "\n");
                            RunCML.shutdown OS.Process.failure)

          val debug =
                case Widget.findNamedOpt optDB
                                         (Styles.OPT_NAMED("debug"))
                                         root of
                     []                   => false
                   | Attrs.AV_Bool b :: _ => b
                   | _                    => raise Fail "cannot happen"

          (* lastPart : string -> string

             lastPart s returns the longest suffix of s consisting of
             digits and letters *)

          fun lastPart s =
                let fun iter(xs, nil)     = implode xs
                      | iter(xs, y :: ys) =
                          if Char.isAlpha y orelse Char.isDigit y
                          then iter(y :: xs, ys)
                          else implode xs
                in iter(nil, rev(explode s)) end

          val name =
                case Widget.findNamedOpt optDB
                                         (Styles.OPT_NAMED("name"))
                                         root of
                     []                  => lastPart prog
                   | Attrs.AV_Str s :: _ => s
                   | _                   => raise Fail "cannot happen"

          val _ =
                XDebug.init(if debug
                            then ["+eXene/pretty"]
                            else ["-/ThreadWatcher/"])

          val xrdStyle  =
                Widget.styleFromXRDB root
                  handle Styles.PRS.BadSpec n =>
                     (TextIO.output(TextIO.stdErr,
                                    prog ^ ": " ^
                                    "bad resource specification: " ^
                                    Int.toString n ^ "\n");
                      Widget.delRoot root; RunCML.shutdown OS.Process.failure)
          val argStyle  =
                Widget.styleFromOptDb(root, optDB)
                  handle Styles.PRS.BadSpec n =>
                     (TextIO.output(TextIO.stdErr,
                                    prog ^ ": " ^
                                    "bad resource specification: " ^
                                    Int.toString n ^ "\n");
                      Widget.delRoot root; RunCML.shutdown OS.Process.failure)
          val mainStyle = Widget.mergeStyles(argStyle, xrdStyle)
    
          val _ =
                if debug
                then app (fn s => TextIO.print(s ^ "\n"))
                         (Styles.stringsFromStyle mainStyle)
                else ()
    
          val styleView =
                Styles.mkView{name    = Styles.styleName[name],
                              aliases = [Styles.styleName["Goodbye"]]}
          val view      = (styleView, mainStyle)
      in loop(root, view);
         W.delRoot root;
         print "\n";
         RunCML.shutdown OS.Process.success
      end

fun main(prog, args) = RunCML.doit(fn () => start(prog, args), NONE)

end;
