/* main.c */

/* Allen Stoughton -- 1994 */

#include <string.h>
#include <stdlib.h>
#include <stddef.h>

#include "interface.h"  /* includes <stdio.h> */

/* declaration of the UNIX System V system call unlink */
int unlink(const char *path);

/* Parameters */

#define InputSuffix "lam"  /* suffix of input files */

/* MaxNumElems to the power MaxTupArity should fit in a signed long int */
#define MaxNumElems 8  /* elements in iota */
#define MaxTupArity 10  /* arity of tuples; equivalently, maximum number of
                           tests */

#define MaxNumFuns 100  /* functions */
#define MaxNumClauses 100  /* clauses in a function definition */
#define MaxType 15  /* of a function; equivalently, maximum number of
                       patterns in a clause of a function definition */
#define MaxNumVars 15  /* variables */
#define MaxNumArgTups 50  /* argument tuples; equivalently, maximum number
                             of test arguments */

/* Declarations */

/* the following two symbols are defined by mach_text_decl.o, which
   is built by Makefile from mach.c; mach_text contains the text of
   mach.c, and mach_text_size is the number of bytes in mach_text */
extern char mach_text[];
extern int mach_text_size;

int line_num = 1;  /* current line number in input file */
char *program;  /* name of executable */
char *output_file;  /* file that C code will be written to */
FILE *output_fp = NULL;  /* corresponding file pointer; see error_exit
                            to see why it's initialized to NULL */

typedef int Tup[MaxTupArity];  /* tuple type */

int tup_arity = 0;  /* tuple arity; must be > 0, eventually */
int num_tups;  /* number of iota tuples; computed by compute_num_tups */
int max_type = 0;  /* largest type of constant or argument tuple */
int num_arg_tups = 0;  /* number of argument tuples */

/* symbol tables */

struct {  /* elements */
  char *string;  /* name */
  int new;  /* just created? */
  int is_constant;  /* set by constants section */
} elems[MaxNumElems];

int num_elems = 0;  /* number of elements */

struct {  /* functions */
  char *string;  /* name */
  int new;  /* just created? */
  int type;  /* always > 0 */
  int is_constant;  /* set by constants section */
} funs[MaxNumFuns];

int num_funs = 0;  /* number of functions */

struct {  /* variables */
  char *string;  /* name */
  int new;  /* just created? */
  int param;  /* parameter number */
} vars[MaxNumVars];

int num_vars = 0;  /* number of variables */

/* functions section */

#define WildCard -1

/* elements of the following array are indices of elems or are WildCard */
int patterns[MaxNumClauses][MaxType];

/* elements of the following array are indices of elems or are of the
   form -(i+1) where 0 <= i < fun_type */
int results[MaxNumClauses];

/* an element i of the following array is non-zero iff clause i of cur_fun
   is redundant */
int redundant[MaxNumClauses];

int num_clauses;  /* number of clauses */
int param;  /* second index to patterns */
int cur_fun;  /* index of funs; used by functions section */
int fun_type;  /* type of cur_fun */

/* an element arr[i][j] of the following structure is non-zero iff
   the ith argument of cur_fun COULD BE the jth element of elems;
   it's a structure so that elements of Possib can be copied by
   assignments and function calls */
typedef struct {
  char arr[MaxType][MaxNumElems];
} Possib;

/* tests section */

struct {
  int type;  /* of tuple components */
  Tup tup;  /* components are indices of elems (if type == 0) or funs
               (if type > 0) */
} arg_tups[MaxNumArgTups];

int arg_tup = 0;  /* index of arg_tups */

Tup result_tup;  /* elements are indices of elems */

/* Auxiliary Functions */

void error_exit(void)
{
  if (output_fp)  /* if the output file has been created, remove it */
    unlink(output_file);
  exit(1);
}

void error(const char *msg)
{
  if (*yytext)
    printf("%s: line %d at \"%s\": %s\n", program, line_num, yytext, msg);
  else
    printf("%s: line %d at end of file: %s\n", program, line_num, msg);
  error_exit();
}

void yyerror(const char *msg)
{
  error(msg);
}

int yywrap(void)
{
  return 1;
}

void indent(int ind)
{
  for (; ind > 0; ind--)
    putc(' ', output_fp);
}

void *safe_malloc(size_t len)
{
  void *p = malloc(len);

  if (!p) {
    printf("%s: storage limit exceeded\n", program);
    error_exit();
  }
  return(p);
}

char *safe_strdup(const char *s)
{
  char *p = safe_malloc(strlen(s) + 1);

  strcpy(p, s);
  return(p);
}

/* returns an index of elems */
int make_elem(char *s)
{
  int elem;

  for (elem = 0; elem < num_elems; elem++)
    if (strcmp(s, elems[elem].string) == 0) {
      elems[elem].new = 0;
      return elem;
    }
  if (num_elems == MaxNumElems)
    error("too many element names used");
  elems[num_elems].string = safe_strdup(s);
  elems[num_elems].new = 1;
  elems[num_elems].is_constant = 0;
  return num_elems++;
}

/* returns an index of funs */
int make_fun(char *s)
{
  int fun;

  for (fun = 0; fun < num_funs; fun++)
    if (strcmp(s, funs[fun].string) == 0) {
      funs[fun].new = 0;
      return fun;
    }
  if (num_funs == MaxNumFuns)
    error("too many function names used");
  funs[num_funs].string = safe_strdup(s);
  funs[num_funs].new = 1;
  funs[num_funs].is_constant = 0;
  return num_funs++;
}

/* returns an index of vars */
int make_var(char *s)
{
  int var;

  for (var = 0; var < num_vars; var++)
    if (strcmp(s, vars[var].string) == 0) {
      vars[var].new = 0;
      return var;
    }
  if (num_vars == MaxNumVars)
    error("too many variable names used");
  vars[num_vars].string = safe_strdup(s);
  vars[num_vars].new = 1;
  return num_vars++;
}

void free_vars(void)
{
  int var;

  for (var = 0; var < num_vars; var++)
    free(vars[var].string);
  num_vars = 0;
}

/* convert tuple of elements, treated as a base num_elems number,
   to an int code between 0 and num_tups - 1; works even when
   num_elems == 1 */
int tup_to_code(Tup tup)
{
  int comp, code;

  for (code = tup[0], comp = 1 ; comp < tup_arity; comp++)
    code = code * num_elems + tup[comp];
  return code;
}

/* convert the code of a tuple back to the tuple */
void code_to_tup(int code, Tup tup)
{
  int comp;

  for (comp = tup_arity - 1; comp > 0; comp--) {
    tup[comp] = code % num_elems; code /= num_elems;
  }
  tup[0] = code;
}

/* Action Functions */

void iota_sect_act(void)
{
  int elem;

  if (num_elems == 0)
    error("no elements specified");
  fprintf(output_fp, "typedef enum {\n");
  for (elem = 0; elem < num_elems; elem++) {
    fprintf(output_fp, "  Elem_%s", elems[elem].string);
    if (elem < num_elems - 1)
      putc(',', output_fp);
    putc('\n', output_fp);
  }
  fprintf(output_fp, "} Iota;\n\n");
  fprintf(output_fp, "#define NumElems %d\n\n", num_elems);
  fprintf(output_fp, "const char *iota_strings[NumElems] = {\n");
  for (elem = 0; elem < num_elems; elem++) {
    fprintf(output_fp, "  \"%s\"", elems[elem].string);
    if (elem < num_elems - 1)
      putc(',', output_fp);
    putc('\n', output_fp);
  }
  fprintf(output_fp, "};\n\n");
}

void elem_act(int elem)
{
  if (!elems[elem].new)
    error("element repeated");
}

/* used by fun_act; generate body of function definition */
void fun_body(Possib pos, int ind)
{
  int clause, param, elem; Possib newpos;

  for (clause = 0; clause < num_clauses; clause++) {  /* look for first
                                                         matchable clause */
    for (param = 0;
         param < fun_type &&
         (patterns[clause][param] == WildCard ||
          pos.arr[param][patterns[clause][param]]);
         param++)
      ;
    if (param == fun_type)
      break;
  }
  if (clause == num_clauses)  /* no matchable clauses */
    error("preceding function definition is incomplete");
  /* matchable clause found */
  for (param = 0; param < fun_type; param++) {  /* will clause definitely
                                                   match? */
    if (patterns[clause][param] != WildCard) {
      for (elem = 0;
           elem < num_elems &&
           (elem == patterns[clause][param] || !pos.arr[param][elem]);
           elem++)
        ;
      if (elem != num_elems)
        break;
    }
  }
  if (param == fun_type) {  /* clause will definitely match */
    redundant[clause] = 0;  /* this clause is not redundant */
    indent(ind);
    fprintf(output_fp, "return ");
    if (results[clause] >= 0)
      fprintf(output_fp, "Elem_%s;\n", elems[results[clause]].string);
    else
      fprintf(output_fp, "x%d;\n", -results[clause] - 1);
  } else {  /* clause won't necessarily be matched at pattern param */
    newpos = pos;  /* argument param is definitely patterns[clause][param] */
    for (elem = 0; elem < num_elems; elem++)
      newpos.arr[param][elem] = 0;
    newpos.arr[param][patterns[clause][param]] = 1;
    indent(ind);
    fprintf(output_fp, "if (x%d == Elem_%s)\n",
            param, elems[patterns[clause][param]].string);
    fun_body(newpos, ind + 2);
    newpos = pos;  /* argument param is not patterns[clause][param] */
    newpos.arr[param][patterns[clause][param]] = 0;
    indent(ind);
    fprintf(output_fp, "else\n");
    fun_body(newpos, ind + 2);
  }
}

void fun_act(void)
{
  int clause, param, elem;  Possib pos;

  if (num_clauses == 0)
    error("no clauses in preceding function definition");
  fprintf(output_fp, "Iota Fun_%s(", funs[cur_fun].string);
  for (param = 0; param < fun_type; param++) {
    fprintf(output_fp, "Iota x%d", param);
    if (param < fun_type - 1)
      fprintf(output_fp, ", ");
  }
  fprintf(output_fp, ")\n{\n");
  for (param = 0; param < fun_type; param++)
    for (elem = 0; elem < num_elems; elem++)
      pos.arr[param][elem] = 1;
  for (clause = 0; clause < num_clauses; clause++)
    redundant[clause] = 1;
  fun_body(pos, 2);
  for (clause = 0; clause < num_clauses; clause++)
    if (redundant[clause])
      error("redundant clause in preceding function definition");
  fprintf(output_fp, "}\n\n");
}

void fun_name_act(int fun)
{
  if (!funs[fun].new)
    error("redefinition of function");
  cur_fun = fun;
  num_clauses = 0;
  param = 0;
}

void clause_act(void)
{
  num_clauses++;
  param = 0;
  free_vars();
}

void pat_elem_act(int elem)
{
  if (num_clauses == MaxNumClauses)
    error("too many clauses in function definition");
  if (param == (num_clauses == 0 ? MaxType : fun_type))
    error("too many patterns in clause");
  if (elems[elem].new)
    error("undefined element");
  patterns[num_clauses][param] = elem;
  param++;
}

void pat_var_act(int var)
{
  if (num_clauses == MaxNumClauses)
    error("too many clauses in function definition");
  if (param == (num_clauses == 0 ? MaxType : fun_type))
    error("too many patterns in clause");
  if (!vars[var].new)
    error("variable re-bound in clause");
  vars[var].param = param;
  patterns[num_clauses][param] = WildCard;
  param++;
}

void pat_wildcard_act(void)
{
  if (num_clauses == MaxNumClauses)
    error("too many clauses in function definition");
  if (param == (num_clauses == 0 ? MaxType : fun_type))
    error("too many patterns in clause");
  patterns[num_clauses][param] = WildCard;
  param++;
}

void pats_end_act(void)
{
  if (num_clauses == MaxNumClauses)
    error("too many clauses in function definition");
  if (num_clauses == 0) {
    if (param == 0)
      error("clause has no patterns");
    funs[cur_fun].type = fun_type = param;
  } else if (param != fun_type)
    error("too few patterns in clause");
}

void result_elem_act(int elem)
{
  if (elems[elem].new)
    error("undefined element");
  results[num_clauses] = elem;
}

void result_var_act(int var)
{
  if (vars[var].new)
    error("unbound variable");
  results[num_clauses] = -(vars[var].param + 1);
}

void con_elem_act(int elem)
{
  if (elems[elem].new)
    error("undefined element");
  if (elems[elem].is_constant)
    error("constant repeated");
  elems[elem].is_constant = 1;
}

void con_fun_act(int fun)
{
  if (funs[fun].new)
    error("undefined function");
  if (funs[fun].is_constant)
    error("constant repeated");
  funs[fun].is_constant = 1;
  if (funs[fun].type > max_type)
    max_type = funs[fun].type;
}

void tests_sect_act(void)
{
  if (tup_arity == 0)
    error("no tests specified");
}

void test_act(void)
{
  tup_arity++;
  arg_tup = 0;
}

void test_arg_elem_act(int elem)
{
  if (tup_arity == MaxTupArity)
    error("too many tests");
  if (arg_tup == (tup_arity == 0 ? MaxNumArgTups : num_arg_tups))
    error("too many test arguments");
  if (elems[elem].new)
    error("undefined element");
  if (tup_arity == 0)
    arg_tups[arg_tup].type = 0;
  else if (arg_tups[arg_tup].type != 0)
    error("test argument has incorrect type");
  arg_tups[arg_tup].tup[tup_arity] = elem;
  arg_tup++;
}

void test_arg_fun_act(int fun)
{
  if (tup_arity == MaxTupArity)
    error("too many tests");
  if (arg_tup == (tup_arity == 0 ? MaxNumArgTups : num_arg_tups))
    error("too many test arguments");
  if (funs[fun].new)
    error("undefined function");
  if (tup_arity == 0) {
    arg_tups[arg_tup].type = funs[fun].type;
    if (funs[fun].type > max_type)
      max_type = funs[fun].type;
  } else if (arg_tups[arg_tup].type != funs[fun].type)
    error("test argument has incorrect type");
  arg_tups[arg_tup].tup[tup_arity] = fun;
  arg_tup++;
}

void test_arg_seq_end_act(void)
{
  if (tup_arity == MaxTupArity)
    error("too many tests");
  if (tup_arity == 0)
    num_arg_tups = arg_tup;
  else if (arg_tup != num_arg_tups)
    error("too few test arguments");
}

void test_result_act(int elem)
{
  if (elems[elem].new)
    error("undefined element");
  result_tup[tup_arity] = elem;
}

/* Additional Code Generation Functions */

/* used by more_code */
void compute_num_tups(void)
{
  int comp; long num;

  /* compute num_elems to the power tup_arity */
  for (num = 1, comp = 0; comp < tup_arity; comp++)
    num *= num_elems;
  if (num > 32767)  /* too large for a 16 bit signed int? */
    error("too many iota tuples");
  num_tups = num;
}

/* used by state_def and closure_arg_tup */
char *arg_tup_var_name(int arg_tup)
{
  char var_name[20];  /* more than enough space */

  sprintf(var_name, "x%d", arg_tup);
  return safe_strdup(var_name);
}

/* used by more_code; generate machine state definition */
void state_def(void)
{
  int code, elem, comp, arg_tup, size;
  Tup tup;
  char **rel;  /* relation: array of strings indexed by tuple codes */

  size = 0;
  rel = safe_malloc(num_tups * sizeof(char *));
  for (code = 0; code < num_tups; code++)
    rel[code] = NULL;  /* tuple not in stage 0 */
  for (elem = 0; elem < num_elems; elem++)
    if (elems[elem].is_constant) {
      for (comp = 0; comp < tup_arity; comp++)
        tup[comp] = elem;
      if (!(rel[code = tup_to_code(tup)])) {
        rel[code] = elems[elem].string;
        size++;
      }
    }
  for (arg_tup = 0; arg_tup < num_arg_tups; arg_tup++)
    if (arg_tups[arg_tup].type == 0) {
      if (!(rel[code = tup_to_code(arg_tups[arg_tup].tup)])) {
        rel[code] = arg_tup_var_name(arg_tup);
        size++;
      }
    }

  fprintf(output_fp, "struct {\n");
  fprintf(output_fp, "  int stage;\n");
  fprintf(output_fp, "  int size;\n");
  fprintf(output_fp, "  int outer_code;\n");
  fprintf(output_fp, "  struct {\n");
  fprintf(output_fp, "    Tup tup;\n");
  fprintf(output_fp, "    enum { No = 0, Next = 01, YesNew = 02, ");
  fprintf(output_fp, "YesOld = 04, Yes = 06 } memb;\n");
  fprintf(output_fp, "    int stage;\n");
  fprintf(output_fp, "    int size;\n");
  fprintf(output_fp, "    const char *string;\n");
  fprintf(output_fp, "    int num_args;\n");
  fprintf(output_fp, "    int args[MaxType];\n");
  fprintf(output_fp, "  } rel[NumTups];\n");
  fprintf(output_fp, "} state = {\n");
  fprintf(output_fp, "  0,\n");
  fprintf(output_fp, "  %d,\n", size);
  fprintf(output_fp, "  0,\n");
  fprintf(output_fp, "  {\n");
  for (code = 0; code < num_tups; code++) {
    fprintf(output_fp, "    { { ");
    code_to_tup(code, tup);
    for (comp = 0; comp < tup_arity; comp++) {
      fprintf(output_fp, "Elem_%s", elems[tup[comp]].string);
      if (comp < tup_arity - 1)
        fprintf(output_fp, ", ");
    }
    fprintf(output_fp, " }, ");
    if (rel[code])
      fprintf(output_fp, "YesNew, 0, 1, \"%s\", 0", rel[code]);
    else
      fprintf(output_fp, "No");
    fprintf(output_fp, " }");
    if (code < num_tups - 1)
      putc(',', output_fp);
    putc('\n', output_fp);
  }
  fprintf(output_fp, "  }\n");
  fprintf(output_fp, "};\n\n");

  free(rel);
}

/* used by closure_body; generate code to close under a constant */
void closure_con(int type, int fun, int ind)
{
  int arg, comp;

  for (comp = 0; comp < tup_arity; comp++) {
    indent(ind);
    fprintf(output_fp, "code = ");
    if (comp != 0)
      fprintf(output_fp, "code * NumElems + ");
    fprintf(output_fp, "Fun_%s(", funs[fun].string);
    for (arg = 0; arg < type; arg++) {
      fprintf(output_fp, "state.rel[n%d].tup[%d]", arg, comp);
      if (arg < type - 1)
        fprintf(output_fp, ", ");
    }
    fprintf(output_fp, ");\n");
  }
  indent(ind);
  fprintf(output_fp, "size = 1;\n");
  for (arg = 0; arg < type; arg++) {
    indent(ind);
    fprintf(output_fp, "size += state.rel[n%d].size;\n", arg);
  }
  indent(ind);
  fprintf(output_fp, "if (state.rel[code].memb == No ||\n");
  indent(ind + 4);
  fprintf(output_fp, "state.rel[code].memb == Next && ");
  fprintf(output_fp, "size < state.rel[code].size) {\n");
  indent(ind + 2);
  fprintf(output_fp, "state.rel[code].memb = Next;\n");
  indent(ind + 2);
  fprintf(output_fp, "state.rel[code].stage = state.stage + 1;\n");
  indent(ind + 2);
  fprintf(output_fp, "state.rel[code].size = size;\n");
  indent(ind + 2);
  fprintf(output_fp, "state.rel[code].string = \"%s\";\n", funs[fun].string);
  indent(ind + 2);
  fprintf(output_fp, "state.rel[code].num_args = %d;\n", type);
  for (arg = 0; arg < type; arg++) {
    indent(ind + 2);
    fprintf(output_fp, "state.rel[code].args[%d] = n%d;\n", arg, arg);
  }
  indent(ind);
  fprintf(output_fp, "}\n");
}

/* used by closure_body; generate code to close under an argument tuple */
void closure_arg_tup(int type, int arg_tup, int ind)
{
  int arg, comp;

  for (comp = 0; comp < tup_arity; comp++) {
    indent(ind);
    fprintf(output_fp, "code = ");
    if (comp != 0)
      fprintf(output_fp, "code * NumElems + ");
    fprintf(output_fp, "Fun_%s(", funs[arg_tups[arg_tup].tup[comp]].string);
    for (arg = 0; arg < type; arg++) {
      fprintf(output_fp, "state.rel[n%d].tup[%d]", arg, comp);
      if (arg < type - 1)
        fprintf(output_fp, ", ");
    }
    fprintf(output_fp, ");\n");
  }
  indent(ind);
  fprintf(output_fp, "size = 1;\n");
  for (arg = 0; arg < type; arg++) {
    indent(ind);
    fprintf(output_fp, "size += state.rel[n%d].size;\n", arg);
  }
  indent(ind);
  fprintf(output_fp, "if (state.rel[code].memb == No ||\n");
  indent(ind + 4);
  fprintf(output_fp, "state.rel[code].memb == Next && ");
  fprintf(output_fp, "size < state.rel[code].size) {\n");
  indent(ind + 2);
  fprintf(output_fp, "state.rel[code].memb = Next;\n");
  indent(ind + 2);
  fprintf(output_fp, "state.rel[code].stage = state.stage + 1;\n");
  indent(ind + 2);
  fprintf(output_fp, "state.rel[code].size = size;\n");
  indent(ind + 2);
  fprintf(output_fp,
          "state.rel[code].string = \"%s\";\n",
          arg_tup_var_name(arg_tup));
  indent(ind + 2);
  fprintf(output_fp, "state.rel[code].num_args = %d;\n", type);
  for (arg = 0; arg < type; arg++) {
    indent(ind + 2);
    fprintf(output_fp, "state.rel[code].args[%d] = n%d;\n", arg, arg);
  }
  indent(ind);
  fprintf(output_fp, "}\n");
}

/* used by closure_def; generate body of closure function definition */
void closure_body(int type, int ind)
{
  int fun, arg_tup, arg;

  indent(ind);
  fprintf(output_fp,
          "for (n%d = %s; n%d < NumTups; n%d++)\n",
          type - 1, type == 1 ? "state.outer_code" : "0", type - 1, type - 1);
  indent(ind + 2);
  fprintf(output_fp, "if (state.rel[n%d].memb & Yes) {\n", type - 1);
  if (type == 1) {
    indent(ind + 4);
    fprintf(output_fp, "if (checkpoint_requested) {\n");
    indent(ind + 6);
    fprintf(output_fp, "state.outer_code = n0;\n");
    indent(ind + 6);
    fprintf(output_fp, "checkpoint();\n");
    indent(ind + 4);
    fprintf(output_fp, "}\n");
  }
  for (fun = 0; fun < num_funs; fun++)
    if (funs[fun].is_constant && funs[fun].type == type)
      break;
  for (arg_tup = 0; arg_tup < num_arg_tups; arg_tup++)
    if (arg_tups[arg_tup].type == type)
      break;
  if (fun != num_funs || arg_tup != num_arg_tups) {
    indent(ind + 4);
    fprintf(output_fp, "if ((");
    for (arg = 0; arg < type; arg++) {
      fprintf(output_fp, "state.rel[n%d].memb", arg);
      if (arg < type - 1)
        fprintf(output_fp, " | ");
    }
    fprintf(output_fp, ") & YesNew) {\n");
    for (; fun < num_funs; fun++)
      if (funs[fun].is_constant && funs[fun].type == type)
        closure_con(type, fun, ind + 6);
    for (; arg_tup < num_arg_tups; arg_tup++)
      if (arg_tups[arg_tup].type == type)
        closure_arg_tup(type, arg_tup, ind + 6);
    indent(ind + 4);
    fprintf(output_fp, "}\n");
  }
  if (type != max_type)
    closure_body(type + 1, ind + 4);
  indent(ind + 2);
  fprintf(output_fp, "}\n");
}

/* used by more_code; generate closure function definition */
void closure_def(void)
{
  int arg;

  fprintf(output_fp, "void closure(void)\n");
  fprintf(output_fp, "{\n");
  if (max_type > 0) {
    fprintf(output_fp, "  int code, size;\n");
    fprintf(output_fp, "  int ");
    for (arg = 0; arg < max_type; arg++) {
      fprintf(output_fp, "n%d", arg);
      if (arg < max_type - 1)
        fprintf(output_fp, ", ");
    }
    fprintf(output_fp, ";\n\n");
    closure_body(1, 2);
  }
  fprintf(output_fp, "}\n\n");
}

void more_code(void)
{
  compute_num_tups();

  fprintf(output_fp, "#define TupArity %d\n\n", tup_arity);

  fprintf(output_fp, "#define NumTups %d\n\n", num_tups);

  fprintf(output_fp, "#define NumArgTups %d\n\n", num_arg_tups);

  /* since MaxType is an array bound, make sure that it's at least 1 */
  fprintf(output_fp, "#define MaxType %d\n\n",
          max_type > 0 ? max_type : 1);

  fprintf(output_fp, "#define ResultTupCode %d\n\n", tup_to_code(result_tup));

  fprintf(output_fp, "typedef Iota Tup[TupArity];\n\n");

  state_def();

  fprintf(output_fp, "int checkpoint_requested;\n\n");

  fprintf(output_fp, "void checkpoint(void);\n\n");

  closure_def();
}

/* Main Function */

int main(int argc, char **argv)
{
  char *trailing, *s, c;

  program = argv[0];

  /* process argument, opening the input file and creating/truncating
     and opening the output file */
  if (argc < 2) {
    printf("%s: no file specified\n", program);
    return 1;
  }
  if (argc > 2) {
    printf("%s: too many arguments specified\n", program);
    return 1;
  }
  /* make copy of trailing component of filename */
  for (s = argv[1]; *s; s++)
    ;
  for (; s != argv[1] && *(s-1) != '/'; s--)
    ;
  trailing = safe_strdup(s);
  /* does trailing have suffix InputSuffix? */
  for (s = trailing; *s; s++)
    ;
  for (; s != trailing && *s != '.'; s--)
    ;
  if (*s != '.' || strcmp(s+1, InputSuffix) != 0) {
    printf("%s: \"%s\" does not have suffix \"%s\"\n",
           program, argv[1], InputSuffix);
    return 1;
  }
  if (s == trailing) {  /* is root of input file null? */
    printf("%s: (trailing component of) \"%s\" has null root\n",
           program, argv[1]);
    return 1;
  }
  if (!(yyin = fopen(argv[1], "r"))) {  /* open input file */
    printf("%s: unable to read \"%s\"\n", program, argv[1]);
    return 1;
  }
  /* initialize output_file as root.c, where root is the root of
     the input file */
  *(s+1) = '\0';
  output_file = safe_malloc(strlen(trailing) + sizeof "c" + 1);
  strcpy(output_file, trailing);
  strcat(output_file, "c");
  free(trailing);
  /* create/truncate and open output file in current dir */
  if (!(output_fp = fopen(output_file, "w"))) {
    printf("%s: unable to create \"%s\"\n", program, output_file);
    return 1;
  }

  fprintf(output_fp, "/* %s */\n\n", output_file);

  if (yyparse())
    error_exit();

  if (ferror(yyin) ||
      fclose(yyin) == EOF) {
    printf("%s: error reading from \"%s\"\n", program, argv[1]);
    error_exit();
  }

  more_code();

  /* output fixed part of code */
  fwrite(mach_text, mach_text_size, 1, output_fp);

  if (ferror(output_fp) ||
      fclose(output_fp) == EOF) {
    printf("%s: error writing to \"%s\"\n", program, output_file);
    error_exit();
  }
  return 0;
}
