/* mach.c */

/* Allen Stoughton -- 1994 */

/* The fixed part of the programs (closure machines) that Lambda
   generates */

/* Transformed into mach_text_decl.o and linked together with main.o
   by Makefile */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stddef.h>
#include <signal.h>

/* declarations of the UNIX System V system calls that are used along with
   signal to implement checkpointing */
unsigned int alarm(unsigned int seconds);
int rename(const char *path1, const char *path2);

#define CheckpointFreq (10 * 60)  /* frequency in seconds of checkpointing */
#define CheckpointSuffix "LAM"  /* suffix of checkpoint files */

char *program;  /* name of executable */
char *checkpoint_file;  /* name of checkpoint file */
char *temp_checkpoint_file;  /* name of temporary checkpoint file */
int verbose = 0;  /* should output be verbose? */

void indent(int ind)
{
  for (; ind > 0; ind--)
    putchar(' ');
}

void *safe_malloc(size_t len)
{
  void *p = malloc(len);

  if (!p) {
    fprintf(stderr, "%s: storage limit exceeded\n", program);
    exit(1);
  }
  return(p);
}

void print_tup(Tup tup)
{
  int comp;

  putchar('<');
  for (comp = 0; comp < TupArity; comp++) {
    printf("%s", iota_strings[tup[comp]]);
    if (comp < TupArity - 1)
      putchar(' ');
  }
  putchar('>');
}

void print_rel(void)
{
  int code;

  for (code = 0; code < NumTups; code++)
    if (state.rel[code].memb & Yes) {
      print_tup(state.rel[code].tup);
      if (verbose)
        printf("  %d", state.rel[code].stage);
      putchar('\n');
    }
}

void print_rel_comp(void)
{
  int code;

  for (code = 0; code < NumTups; code++)
    if (state.rel[code].memb == No) {
      print_tup(state.rel[code].tup);
      putchar('\n');
    }
}

/* used by print_term */
void pr_term(int code, int ind, int offset)
{
  int arg;

  indent(ind);
  printf("%s", state.rel[code].string);
  if (verbose) {
    indent(offset - strlen(state.rel[code].string));
    print_tup(state.rel[code].tup);
  }
  putchar('\n');
  for (arg = 0; arg < state.rel[code].num_args; arg++)
    pr_term(state.rel[code].args[arg], ind + 4, offset);
}

/* used by print_term */
int max_str_len(int code)
{
  int max, arg, n;

  max = strlen(state.rel[code].string);

  for (arg = 0; arg < state.rel[code].num_args; arg++)
    if ((n = max_str_len(state.rel[code].args[arg])) > max)
      max = n;

  return max;
}

void print_term(int code, int ind)
{
  pr_term(code, ind, max_str_len(code) + 4);
}

int tups_remaining(void)  /* to be considered in outer loop of computation
                             of next stage */
{
  int code, rem;

  for (rem = 0, code = state.outer_code; code < NumTups; code++)
    if (state.rel[code].memb & Yes)
      rem++;
  return rem;
}

int new_tups_found(void)  /* in computation of next stage */
{
  int code, new;

  for (new = 0, code = 0; code < NumTups; code++)
    if (state.rel[code].memb & Next)
      new++;
  return new;
}

void print_status(void)
{
  int rem = tups_remaining(), new = new_tups_found();

  fprintf(stderr,
          "%s: working on stage %d with %d tuple%s out of %d remaining\n",
          program, state.stage + 1, rem, rem == 1 ? "" : "s", state.size);
  fprintf(stderr,
          "  to be considered in outer loop; %d new tuple%s",
          new, new == 1 ? "" : "s");
  fprintf(stderr,
          " found so far\n  (%sincluding result tuple)\n",
          state.rel[ResultTupCode].memb & Next ? "" : "not ");
}

void handler(int sig)
{
  /* Under both ANSI C and UNIX System V, SIGALRM's disposition will
     have been reset by the system to SIG_DFL before handler is
     called.  Thus, if an alarm signal is received before further
     SIGALRM's are ignored, then the program will be killed.  This
     is very unlikely to occur, and is only possible if signals
     are being sent to the program by other processes as well as
     by the alarm clock.  Unfortunately, there isn't any way of
     avoiding this problem under ANSI C (as opposed to System V). */

  signal(SIGALRM, SIG_IGN);
  checkpoint_requested = 1;
}

void checkpoint()
{
  FILE *fp;

  if (!(fp = fopen(temp_checkpoint_file, "w")) ||
      fwrite(&state, sizeof state, 1, fp) != 1 ||
      fclose(fp) == EOF ||
      rename(temp_checkpoint_file, checkpoint_file) == -1) {
    fprintf(stderr, "%s: checkpointing error\n", program);
    exit(1);
  }
  print_status();
  checkpoint_requested = 0;
  signal(SIGALRM, handler);
  alarm(CheckpointFreq);
}

void read_checkpoint(void)
{
  FILE *fp;

  if (!(fp = fopen(checkpoint_file, "r")) ||
      fread(&state, sizeof state, 1, fp) != 1 ||
      getc(fp) != EOF ||
      ferror(fp) ||
      fclose(fp) == EOF) {
    fprintf(stderr, "%s: error reading checkpoint from \"%s\"\n",
            program, checkpoint_file);
    exit(1);
  }
  print_status();
}

int tups_were_created(void)  /* during last execution of closure */
{
  int code, size;

  size = state.size;
  for (code = 0; code < NumTups; code++) {
    switch (state.rel[code].memb) {
    case Next:
      state.rel[code].memb = YesNew;
      size++;
      break;
    case YesNew:
      state.rel[code].memb = YesOld;
      break;
    default:
      break;
    }
  }
  if (size > state.size) {
    state.size = size;
    state.stage++;
    return 1;
  } else
    return 0;
}

void print_outcome(void)
{
  int arg_tup;

  printf("%s\n\n", program);
  printf("Stage: %d\n\n", state.stage);
  if (state.rel[ResultTupCode].memb & Yes) {
    printf("Term:\n\n");
    if (NumArgTups == 0)
      print_term(ResultTupCode, 0);
    else {
      printf("lambda ");
      for (arg_tup = 0; arg_tup < NumArgTups; arg_tup++) {
        printf("x%d", arg_tup);
        if (arg_tup < NumArgTups - 1)
          putchar(' ');
      }
      printf(".\n");
      print_term(ResultTupCode, 4);
    }
  } else {
    printf("Relation (%d element%s):\n\n",
           state.size, state.size == 1 ? "" : "s");
    print_rel();
    printf("\nRelation complement (%d element%s):\n\n",
           NumTups - state.size, NumTups - state.size == 1 ? "" : "s");
    print_rel_comp();
  }
}

void usage_error(void)
{
  fprintf(stderr, "usage: %s [-lv]\n", program);
  exit(1);
}

int main(int argc, char **argv)
{
  int load_checkpoint = 0;  /* should checkpoint be loaded? */
  char *s, **a;

  program = argv[0];

  checkpoint_file =
    safe_malloc(strlen(program) + 1 + sizeof CheckpointSuffix + 1);
  strcpy(checkpoint_file, argv[0]);
  strcat(checkpoint_file, ".");
  strcat(checkpoint_file, CheckpointSuffix);
  temp_checkpoint_file = safe_malloc(strlen(checkpoint_file) + 1 + 1);
  strcpy(temp_checkpoint_file, checkpoint_file);
  strcat(temp_checkpoint_file, "+");

  for (a = argv + 1; *a; a++) {  /* process options */
    s = *a;
    if (*s != '-')
      usage_error();
    for (s++; *s; s++)
      switch (*s) {
      case 'l':
        load_checkpoint = 1;
        break;
      case 'v':
        verbose = 1;
        break;
      default:
        usage_error();
      }
  }

  if (load_checkpoint)
    read_checkpoint();
  else {
    if (state.rel[ResultTupCode].memb & Yes) {
      print_outcome();
      return 0;
    }
  }

  checkpoint_requested = 0;
  signal(SIGALRM, handler);
  alarm(CheckpointFreq);

  closure();
  while (tups_were_created() && state.rel[ResultTupCode].memb == No) {
    state.outer_code = 0;
    closure();
  }
  print_outcome();

  return 0;
}
