(* str-edit.sml
 *
 * COPYRIGHT (c) 1991 by AT&T Bell Laboratories  See COPYRIGHT file for details.
 *
 * String edit widget.
 * 
 * Modified Aug 2005 to interact with Shell focusable interface, ddeboer.
 *)

signature STREDIT = 
  sig

    structure W : WIDGET
    
    type str_edit
  
    val mkStrEdit : W.root -> {
      foregrnd : W.EXB.color option,
      backgrnd : W.EXB.color option,
      initval : string,
      minlen : int
    } -> str_edit

    val setString : str_edit -> string -> unit
    val getString : str_edit -> string
    val shiftWin  : str_edit -> int -> unit
    val widgetOf : str_edit -> W.widget

    (* added ddeboer *)
    val takeFocus : str_edit * W.EXB.XTime.time -> unit
    val focusableOf : str_edit -> Shell.focusable
    
  end (* STREDIT *)

structure StrEdit : STREDIT =
  struct

    structure EXB = EXeneBase
    structure W = Widget
    (* added ddeboer *)
    structure M = Shell
    
    open CML Geometry EXeneWin Interact Drawing

    val min = Int.min
    val max = Int.max

    datatype rqst
      = GetString
      | GetBounds
      | SetString of string
      | ShiftWin of int
      | DoRealize of {
          env : Interact.in_env,
          win : EXB.window,
          sz : size
        }
    (* added ddeboer spring 2005 *)
      | TakeFocus of EXB.XTime.time
      
    datatype reply
      = Bnds of W.bounds
      | Str of string

    datatype input
      = MoveC of int
      | Insert of char
      | Erase
      | Kill
      (* added by ddeboer *)
      | SelStart of int
      | SelEnd   of int

  fun keyP (k, inputc, widChan) = let
    val lookup = lookupString defaultTranslation
    fun isErase c = (c = #"\^H")
    fun isKill c = (c = #"\^X")
(* modified below by ddeboer for moving focus *)
    fun doChars (s,mk,xt) = let
      val _ = TextIO.print ("StrEdit: '"^s^"'\n")
      val slen = size s
      fun doChar i =
        if i = slen then ()
        else let
          val c = String.sub(s,i)
        in
(* NOTE: 0xa0 = (ord #" " + 128) *)
          if ((c >= #" ") andalso ((c <= #"~") orelse (Char.ord c >= 0xa0)))
        then (send (inputc, Insert c); doChar (i+1))
          else if isErase c
        then (send (inputc, Erase); doChar (i+1))
          else if isKill c
        then (send (inputc, Kill); doChar (i+1))
          else if ((c = #"\t") andalso (shiftIsSet mk))
        then (send(widChan,M.Previous xt))
          else if (c = #"\t")
        then (send(widChan,M.Next xt))
          else doChar(i+1)
        end
    in
      doChar 0
    end

    fun loop () =
      case msgBodyOf (sync k) of 
        KEY_Press (ks,mk,xt) => (
          (doChars ((lookup (ks,mk,xt)),mk,xt) 
            handle KeysymNotFound => (case ks of KEYSYM(i) => TextIO.print ("KEYSYM("^(Int.toString i)^") not found.\n")));
          loop ()
        )
      | _ => loop ()
  in
    loop ()
  end

  (* modified ddeboer for assuming focus and setting selection; was: 
  fun mseP (m, mchan, pttopos, widChan) = let
    val waitUp = whileMouseState mbutSomeSet
    val mevt = wrap (m, fn evt => msgBodyOf evt)
    fun loop () =
      case msgBodyOf (sync m) of 
        MOUSE_FirstDown {pt,but,time,...} => (
          send (widChan, (M.Assign time));
          send (mchan, MoveC (pttopos pt));
          waitUp (mkButState [but], mevt);
          loop ()
        )
      | _ => loop ()
  in
    loop ()
  end*)
  fun mseP (m, mchan, pttopos, widChan) = let
    fun loopup () =
      case msgBodyOf (sync m) of 
        MOUSE_FirstDown {pt,but,time,...} => (
          send (widChan, (M.Assign time));
          send (mchan, MoveC (pttopos pt));
          send (mchan, SelStart (pttopos pt));
          loopdn ()
        )
      | _ => loopup () (* logically this case should not occur. *)
    and loopdn () =
      case msgBodyOf (sync m) of
        MOUSE_LastUp {pt,but,time,...} => (
          send (mchan, MoveC (pttopos pt));
          send (mchan, SelEnd (pttopos pt));
          loopup ()
        )
      | _ => loopdn ()
  in
    loopup ()
  end
  
  val dfltMinchars = 4

  datatype str_edit = StrEdit of (W.widget * rqst chan * reply chan
    (* added ddeboer *) * M.focusable_msg chan)

  fun mkStrEdit root {
    foregrnd : color option,
    backgrnd : color option,
    initval : string,
    minlen : int
  } =
  let

    val minchars = max(minlen, dfltMinchars)
    val (bndf, pttopos, realize) = TxtWin.mkTxtWin root (foregrnd, backgrnd)
    val reqChan = channel () and repChan = channel ()
    val focChan : M.focusable_msg chan = channel ()
    val inputc = channel ()
    val SIZE{wid=minlen,...} = bndf minchars

    fun getbnds slen = let
      val SIZE{wid,ht} = bndf (max(minchars,slen))
      val x_dim = W.DIM{base=0, incr=1, min=minlen, nat=wid, max=NONE}
    in
      {x_dim=x_dim, y_dim= W.fixDim ht}
    end

    fun initOff (slen, winlen) =
      if slen <= winlen then 0
      else slen - (winlen div 2)

    fun realizeStrEdit {env=InEnv{m,k,ci,co}, win, sz} initStr = let
      val my_win = win
      val {set_size, set_cur_pos, set_cursor, 
        insert, reset, deletec, set_highlight} = realize (my_win, sz)

      fun main winLen me = let

        fun isCurVisible (_,pos,woff,ss,se) =
          (woff <= pos) andalso (pos <= woff+winLen)

        fun redraw (me as (str,pos,woff,ss,se)) = (
          reset ();
          insert (ExtStr.es_subs(str,woff,winLen));
          set_highlight (ss,se);
          if isCurVisible me then (
            set_cur_pos (pos - woff);
            set_cursor true
          )
          else ()
        )

        fun rightShift (v, me as (str,pos,woff,ss,se)) =
          if v = 0 then me
          else let
            val me' = (str, pos, woff + v, ss, se)
          in
            if v = 1 then (
              set_cursor false;
              set_cur_pos 1;
              deletec (ExtStr.es_subs(str,woff+winLen,1) handle ExtStr.BadIndex _ => "");
              if isCurVisible me' then (
                set_cur_pos (pos - woff - 1);
                set_cursor true
              )
              else ()
            )
            else redraw me';
            me'
          end

        fun leftShift (v, me as (str,pos,woff,ss,se)) =
          if v = 0 then me
          else let
            val me' = (str, pos, woff - v, ss, se)
          in
            if v = 1 then (
              set_cursor false;
              set_cur_pos 0;
              insert (ExtStr.es_subs(str,woff-1,1));
              if isCurVisible me' then (
                set_cur_pos (pos - woff + 1);
                set_cursor true
              )
              else ()
            )
            else redraw me';
            me'
          end

        fun shiftWin (v, me as (str,_,woff,ss,se)) =
          if v <= 0 then (
            if woff = 0 then W.ringBell root 0 else ();
            leftShift (min(~v,woff),me)
          )
          else rightShift (min(v,(ExtStr.es_len str)-woff), me)

        fun mkCurVis (me as (str, pos, woff, ss, se)) =
          if isCurVisible me then me
          else if pos < woff then 
            leftShift (woff-max(0,pos - (winLen div 2)),me)
          else
            rightShift (pos - (winLen div 2) - woff,me)
        
        (* added ddeboer, to calculate bounds rectangle from string length *)
        fun bndsRect slen = 
            let
            val SIZE{wid,ht} = bndf (max(minchars,slen))
            in Geometry.RECT{x=0,y=0,wid=wid,ht=ht} end
            
        fun insertc (c, me as (str, pos, woff,ss,se)) =
          if pos - woff = winLen then
            let
              val woff' = max(pos-1,pos+1-winLen)
              val me' = (ExtStr.es_ins (str,pos,c),pos+1,woff',ss,se)
            in
              (* ddeboer, modified following line to send requested size. *)
              if ExtStr.es_len str = winLen then sync(co (CO_ResizeReq (bndsRect winLen))) else ();
              redraw me';
              me'
            end
          else (
          (* ddeboer, modified following line to send requested size. *)
            if ExtStr.es_len str = winLen then sync(co (CO_ResizeReq (bndsRect winLen))) else ();
            insert (String.str c);
            (ExtStr.es_ins (str, pos, c), pos+1,woff,ss,se)
          )

        fun erasec (me as (str, pos, woff, ss, se)) =
          if pos = 0 then (
            W.ringBell root 0;
            me
          )
          else if pos = woff andalso woff > 0 then
            let
              val woff' = max(0,pos+1-winLen)
              val me' = (ExtStr.es_del (str,pos),pos-1,woff',ss,se)
            in
              if ExtStr.es_len str > winLen then sync(co (CO_ResizeReq (bndsRect winLen))) else ();
              redraw me';
              me'
            end
          else (
            if (ExtStr.es_len str <= (winLen+3)) andalso (winLen < ExtStr.es_len str) then 
              sync(co (CO_ResizeReq (bndsRect winLen))) 
            else ();
            deletec (ExtStr.es_subs(str,woff+winLen,1) handle ExtStr.BadIndex _ => "");
            (ExtStr.es_del(str,pos),pos-1,woff,ss,se)
          )

        fun kill (str,_,_,_,_) = let
          val me' = (ExtStr.mkExtStr "", 0, 0, 0, 0)
        in
          if ExtStr.es_len str > winLen then sync(co (CO_ResizeReq (bndsRect winLen))) else ();
          redraw me';
          me'
        end

        fun handleInput (MoveC p,(str,pos,woff,ss,se)) =
          let
            val pos' = min(ExtStr.es_len str,woff+p)
          in
            if pos <> pos' then (
              set_cur_pos (pos' - woff);
              set_cursor true
            )
            else ();
            (str,pos',woff,ss,se)
          end
          | handleInput (Insert c, me) = insertc(c, mkCurVis me)
          | handleInput (Erase, me) = erasec (mkCurVis me)
          | handleInput (Kill, me) = kill me
          (* added ddeboer to handle selections *)
          | handleInput (SelStart i, (str,pos,woff,ss,se)) = (str,pos,woff,i,se)
          | handleInput (SelEnd i, (str,pos,woff,ss,se)) = (str,pos,woff,ss,i)

        fun handleCI (CI_Resize (RECT{wid,ht,...}), (str,pos,_,ss,se)) =
            initMain (SIZE{wid=wid,ht=ht},str,pos,ss,se)
          | handleCI (CI_Redraw _, me) = (redraw me; me)
          (* modified ddeboer for handling focus *)
          | handleCI (CI_FocusIn, me) = (CML.send(focChan,M.FocusIn); me)
          | handleCI (CI_FocusOut, me) = (CML.send(focChan,M.FocusOut); me)
          | handleCI (_,me) = me

        fun handleReq (GetString,me as (str,_,_,_,_)) = 
            (send(repChan, Str (ExtStr.es_gets str));me)
          | handleReq (ShiftWin arg,me as (str,_,_,_,_)) = 
            shiftWin (arg, me)
          | handleReq (GetBounds,me as (str,_,_,_,_)) = 
            (send(repChan, Bnds (getbnds (ExtStr.es_len str)));me)
          | handleReq (SetString s,_) =
            let
              val slen = size s
              val me' = (ExtStr.mkExtStr s, slen, initOff(slen, winLen), 0, 0)
            in
              sync(co (CO_ResizeReq (bndsRect winLen)));
              redraw me';
              me'
            end
          | handleReq (DoRealize _,me) = me
         (* added following, ddeboer spring 2005 *)
          | handleReq (TakeFocus(xt),me) = (EXeneWin.setInputFocus(my_win,xt);me)
        fun loop me =
          loop (select [
            wrap (ci, fn evt => handleCI (msgBodyOf evt,me)),
            wrap (recvEvt reqChan, fn evt => handleReq (evt,me)),
            wrap (recvEvt inputc, fn evt => handleInput (evt,me))
          ])

      in
        loop me
      end

      and initMain (sz,str,pos,ss,se) = let
        val winlen = set_size sz
      in
        main winlen (str, pos, initOff(pos,winlen), 0, 0)
      end

    in
      spawn (fn () => mseP (m, inputc, pttopos, focChan));
      spawn (fn () => keyP (k, inputc, focChan));
      initMain (sz, ExtStr.mkExtStr initStr, size initStr, 0, 0)
    end

    fun initLoop str =
      case recv reqChan of
        GetString => (send(repChan, Str str); initLoop str)
      | GetBounds => (send(repChan, Bnds (getbnds (size str))); initLoop str)
      | SetString str' => initLoop str'
      | DoRealize arg => realizeStrEdit arg str
      | ShiftWin _ => initLoop str
      (* added following line, ddeboer spring 2005 *)
      | TakeFocus(xt) => initLoop str

  in
    spawn (fn () => (initLoop initval;()));
    StrEdit (
      W.mkWidget{
        root=root,
        args= fn () => {background = NONE}, 
        boundsOf = fn () => (
          send (reqChan, GetBounds);
          case recv repChan of
            Bnds b => b
          | Str _ => raise LibBase.Impossible "StrEdit.mkStrEdit"
        ),
        realize = (fn arg => (send (reqChan, DoRealize arg)))
      },
      reqChan,
      repChan,
      focChan (* added by ddeboer *)
    )
  end

  fun widgetOf (StrEdit(widget,_,_,_)) = widget

  fun setString (StrEdit(_,reqc,_,_)) arg = (send (reqc, SetString arg))

  fun shiftWin (StrEdit(_,reqc,_,_)) arg = (send (reqc, ShiftWin arg))

  fun getString (StrEdit(_,reqc,repc,_)) = (
    send (reqc, GetString);
    case recv repc of
      Bnds _ => raise LibBase.Impossible "StrEdit.getString"
    | Str s => s
  )

  (* added ddeboer, spring 2005 *)
  fun takeFocus (StrEdit(_,reqc,_,_),xt) = (send (reqc,TakeFocus(xt)))
  fun focusableOf (StrEdit(widget,reqc,_,fc)) = 
       M.Focusable {focusableEvt=(WidgetBase.wrapQueue (recvEvt fc)),
                     takefocus=(fn xt => (send (reqc,TakeFocus(xt))) )}
end (* StrEdit *)

