(* simple-edit.sml
 * Based on str-edit.sml,
 * COPYRIGHT (c) 1991 by AT&T Bell Laboratories  See COPYRIGHT file for details.
 *
 * Simple string edit widget, spring 2005, Dusty deBoer, Kansas State University.
 *)

signature SIMPLEEDIT = 
  sig

    structure W : WIDGET
    
    type simple_edit
  
    val simpleEdit : (W.root * Widget.view * Widget.arg list) -> 
            string -> simple_edit

    val setString : simple_edit -> string -> unit
    val getString : simple_edit -> string
    val setSelection : simple_edit -> (int * int * W.EXB.XTime.time) -> unit
    val getSelection : simple_edit -> string
    val widgetOf : simple_edit -> W.widget

    val takeFocus : simple_edit * W.EXB.XTime.time -> unit
    val focusableOf : simple_edit -> Shell.focusable
    
  end (* SIMPLEEDIT *)

structure SimpleEdit : SIMPLEEDIT =
  struct

    structure EXB = EXeneBase
    structure W = Widget
    structure S = Shell
    
    open CML Geometry EXeneWin Interact Drawing EXeneBase 
    
    val attrs = [
        ([], Attrs.attr_font,         Attrs.AT_Font,    Attrs.AV_Str "9x15"),
        ([], Attrs.attr_background,   Attrs.AT_Color,   Attrs.AV_Str "white"),
        ([], Attrs.attr_foreground,   Attrs.AT_Color,   Attrs.AV_Str "black") (* ,
        ([Quark.quark "selection"], Attrs.attr_background, Attrs.AT_Color, Attrs.AV_Str "black"),
        ([Quark.quark "selection"], Attrs.attr_foreground, Attrs.AT_Color, Attrs.AV_Str "white") *)
      ]   
      
    val min = Int.min
    val max = Int.max

    datatype rqst
      = GetString
      | GetBounds
      | SetString of string
      | SetSelection of (int * int * W.EXB.XTime.time)
      | GetSelection
      (* | ShiftWin of int *)
      | DoRealize of {
          env : Interact.in_env,
          win : EXB.window,
          sz : size
        }
      | TakeFocus of EXB.XTime.time
      
    datatype reply
      = Bnds of W.bounds
      | BndsExn
      | Str of string

  datatype simple_edit = SimpleEdit of (W.widget * rqst chan * reply chan
        * S.focusable_msg chan)

  fun simpleEdit (root,view as (name,style),args) initval =
    let
    val view as (name,style) = (Styles.extendView (name,"simpleedit"),style)
    val fattrs = W.findAttr (W.attrs(view,attrs,args))
    val font = Attrs.getFont (fattrs Attrs.attr_font)
    val backc = Attrs.getColor (fattrs Attrs.attr_background)
    val forec = Attrs.getColor (fattrs Attrs.attr_foreground)
    val penn = newPen [PV_Foreground forec, PV_Background backc]
    val penb = newPen [PV_Foreground backc, PV_Background backc]
    (* fixme:
    val view = (Styles.extendView (name,"selection"),style)
    val fattrs = W.findAttr (W.attrs(view,attrs,args))
    val sbackc = Attrs.getColor (fattrs Attrs.attr_background)
    val sforec = Attrs.getColor (fattrs Attrs.attr_foreground)
    val pens = newPen [PV_Foreground sforec, PV_Background sbackc] *)
    val pens = newPen [PV_Foreground backc, PV_Background forec]
    val reqChan = channel () and repChan = channel ()
    val focChan : S.focusable_msg chan  = channel ()
    val {ascent=fonta,descent=fontd} = Font.fontHt font
    val fonth = fonta + fontd
    val Font.CharInfo{left_bearing=lb,char_wid=fontw,...} = 
            Font.charInfoOf font (Char.ord #"A")
    fun bound (x,y,z) = max(x,min(y,z))
    fun pttopos (str,PT{x,y}) = bound(0, x div fontw, String.size str)
    fun getbnds slen = 
      let
      val wid = slen*fontw
      in {x_dim=W.DIM{base=0,incr=1,min=wid,nat=wid,max=NONE},y_dim=W.fixDim fonth} end                 
    fun realizeSimpEdit {env=InEnv{m,k,ci,co},win,sz=SIZE{wid,ht}} 
                {str,selpos,sellen,seltime} = 
        let
        val rq = recvEvt reqChan
        val dst = drawableOfWin win
        fun setsel (str,ss,sl) xt =
            let
            val str = String.substring (str,ss,sl)
            in if sl>0 then (case
                (ICCC.acquireSelection (win,ICCC.Sel_PRIMARY,xt,
                (ICCC.convertString str))) of
                 SOME sh => ((ICCC.releaseEvt sh), 
                            (fn () => (ICCC.releaseSelection sh)) ) 
               | NONE    => (alwaysEvt (), (fn () => ()) ))
                else (never, (fn () => ()) ) end
        val (initsre,initsrf) = 
            case seltime of 
              SOME xt => (setsel (str,selpos,sellen) xt) 
            | _ => (never,fn () => ())
        fun redraw {str,selpos=ss,sellen=sl,selre,selrf,wid,ht} =
            let
            val (x,y) = (0,fonta)
            val cx = x+(fontw*ss)
            val se = ss+sl
            val ll = (String.size str)-se
            val (p1,s1) = (PT{x=x,y=y},String.substring (str,0,ss))
            val (p2,s2) = (PT{x=(x+(fontw*ss)),y=y},String.substring (str,ss,sl))
            val (p3,s3) = (PT{x=(x+(fontw*se)),y=y},String.substring (str,se,ll))
            in  (* clearDrawable dst; *)
                fillRect dst pens (RECT{x=0,y=0,wid=wid,ht=ht});
                imageString dst penn font (p1,s1);
                if (sl > 0) 
                 then (imageString dst pens font (p2,s2)) else ();
                imageString dst penn font (p3,s3);
                if (sl = 0) 
                 then drawSeg dst penn (LINE(PT{x=cx,y=0},PT{x=cx,y=fonth})) else ()
            end
        fun handleMse (MOUSE_FirstDown {pt,but,time,...},
                me as {str,selpos,sellen,selrf,selre,wid,ht}) =
            let
            val _ = selrf()
            in send (focChan, (S.Assign time));
               {str=str,selpos=(pttopos (str,pt)),sellen=0,selrf=(fn ()=>()),
                selre=never,wid=wid,ht=ht} end
          | handleMse (MOUSE_LastUp {pt,but,time,...}, {str,selpos=ss,wid,ht,...}) =
            let
            val se = pttopos (str,pt)
            val (ss,sl) = if (se<ss) then (se,ss-se) else (ss,se-ss)
            val (sre,srf) = (setsel (str,ss,sl) time)
            val me' = {str=str,selpos=ss,sellen=sl,selrf=srf,selre=sre,wid=wid,ht=ht}
            in (redraw me'; me') end
          | handleMse (_,me) = me
        fun handleKey (KEY_Press key, me as {str,selpos=ss,sellen=sl,selrf,selre,wid,ht}) = 
            (case key of
                (KEYSYM(65289),_,xt) => (* tab *)
                    (send(focChan,S.Next xt); me)     
              | (KEYSYM(65056),_,xt) => (* shift+tab *)
                    (send(focChan,S.Previous xt); me) 
              | _ => let
                     val s1 = String.substring (str,0,ss)
                     val s2 = lookupString defaultTranslation key
                        handle KeysymNotFound => ""
                     val s3 = String.substring (str,ss+sl,(String.size str)-ss-sl)
                     val (s1,s2,ss) = if (s2 = "\^H") then (* backspace *)
                          (String.substring (s1,0,max((String.size s1)-1,0)),"",max(ss-1,0))
                          else if (s2 = "\^X") then (* kill *)
                          (s1,"",ss) else (s1,s2,ss)
                     val me = {str=(s1^s2^s3),selpos=ss+(String.size s2),sellen=0,
                                selre=never,selrf=(fn ()=>()),wid=wid,ht=ht}
                     val _ = selrf() (* release the current selection if we have one *)
                     in redraw me; me end
            )
          | handleKey (_,me) = me
        fun handleCI (CI_Resize (RECT{wid,ht,...}), 
                {str,selpos,sellen,selre,selrf,wid=ow,ht=oh}) = 
                    let
                    val me' = {str=str,selpos=selpos,sellen=sellen,selre=selre,selrf=selrf,wid=wid,ht=ht}
                    in (redraw me'; me') end
          | handleCI (CI_Redraw _, me) = (redraw me; me)
          | handleCI (CI_FocusIn, me)  = (CML.send(focChan,S.FocusIn); me)
          | handleCI (CI_FocusOut, me) = (CML.send(focChan,S.FocusOut); me)
          | handleCI (_, me) = me
        fun handleReq (GetString, me as {str,...}) = 
                (send(repChan, Str str); me)
          | handleReq (GetSelection, me as {str,selpos=ss,sellen=sl,...}) = 
                (send(repChan, Str (String.substring (str,ss,sl))); me)
          | handleReq (GetBounds, me) = 
                (send(repChan, BndsExn); me) (* bounds function should not be called now *)
          | handleReq (SetString s, {selrf,wid,ht,...}) = 
                let
                val _ = selrf()
                val me' = {str=s,selpos=(String.size s),sellen=0,
                            selrf=(fn ()=>()),selre=never,wid=wid,ht=ht}
                in redraw me'; me' end
          | handleReq (SetSelection (a,b,xt), {str,selrf,wid,ht,...}) =
                let
                val _ = selrf()
                val mx = String.size str
                val a = bound(0,a,mx)
                val (ss,sl) = (a,bound(0,b-a,mx-a))
                val (sre,srf) = (setsel (str,ss,sl) xt)
                val me' = {str=str,selpos=ss,sellen=sl,selre=sre,
                            selrf=srf,wid=wid,ht=ht}
                in redraw me'; me' end
          | handleReq (DoRealize _, me) = 
                (raise W.AlreadyRealized; me)
          | handleReq (TakeFocus(xt), me) = 
                (EXeneWin.setInputFocus(win,xt); me)
        fun handleSelRel {str,selpos=ss,sellen,selrf,selre,wid,ht} =
                let 
                val me' = {str=str,selpos=ss,sellen=0,selre=never,selrf=(fn ()=>()),wid=wid,ht=ht} 
                in redraw me'; me' end
        fun loop (me as {selre,...}) =
          loop (select [
            wrap (m,  fn evt => handleMse (msgBodyOf evt,me)),
            wrap (k,  fn evt => handleKey (msgBodyOf evt,me)),
            wrap (ci, fn evt => handleCI  (msgBodyOf evt,me)),
            wrap (rq, fn evt => handleReq (evt,me)),
            wrap (selre, fn () => (handleSelRel me))
          ])
        in
          loop {str=str,selpos=selpos,sellen=sellen,selre=initsre,selrf=initsrf,wid=wid,ht=ht}
        end
    fun initLoop (me as {str,selpos,sellen,seltime}) =
      case recv reqChan of
        GetString => 
         (send(repChan, Str str); initLoop me)
      | GetSelection => 
         (send(repChan, Str (String.substring (str,selpos,sellen))); initLoop me)
      | GetBounds => 
         (send(repChan, Bnds (getbnds (size str))); initLoop me)
      | SetString str' => 
         (initLoop {str=str',selpos=(String.size str'),sellen=0,seltime=seltime})
      | SetSelection (ss,se,xt) =>
        let
        val mx = String.size str
        val ss = bound(0,ss,mx)
        val sl = bound(0,(se-ss),(mx-ss))
        in (initLoop {str=str,selpos=ss,sellen=sl,seltime=SOME xt} ) end
      | DoRealize arg => 
         (realizeSimpEdit arg me)
      | TakeFocus(xt) => 
         (initLoop me) (* should perhaps set flag for taking focus upon realization. *)
    in
      spawn (fn () => 
              (initLoop {str=initval,selpos=(String.size initval),sellen=0,seltime=NONE};
              ()));
      SimpleEdit (
        W.mkWidget{
          root=root,
          args= fn () => {background = NONE}, 
          boundsOf = fn () => (
            send (reqChan, GetBounds);
            case recv repChan of
              Bnds b => b
            | BndsExn => raise W.BoundsFunctionAlreadyCalled
            | Str _ => raise LibBase.Impossible "StrEdit.mkStrEdit"
          ),
          realize = (fn arg => (send (reqChan, DoRealize arg)))
        },
        reqChan,
        repChan,
        focChan
      )
    end
    
  fun widgetOf (SimpleEdit(widget,_,_,_)) = widget

  fun setString (SimpleEdit(_,reqc,_,_)) arg = (send (reqc, SetString arg))
  fun setSelection (SimpleEdit(_,reqc,_,_)) arg = (send (reqc, SetSelection arg))

  fun getString (SimpleEdit(_,reqc,repc,_)) = (
    send (reqc, GetString);
    case recv repc of
      Str s => s
    | _ => raise LibBase.Impossible "SimpleEdit.getString"
  )
  fun getSelection (SimpleEdit(_,reqc,repc,_)) = (
    send (reqc, GetSelection);
    case recv repc of
      Str s => s
    | _ => raise LibBase.Impossible "SimpleEdit.getSelection"
  )
  (* added ddeboer, spring 2005 *)
  fun takeFocus (SimpleEdit(_,reqc,_,_),xt) = (send (reqc,TakeFocus(xt)))
  fun focusableOf (SimpleEdit(widget,reqc,_,fc)) = 
        S.Focusable {focusableEvt=(WidgetBase.wrapQueue (recvEvt fc)),
                     takefocus=(fn xt => (send (reqc,TakeFocus(xt))) )}
end (* SimpleEdit *)

