(* selection-mgr.sml
 *
 * (c) 2005 Dusty deBoer, Kansas State University, ddeboer@cis.ksu.edu
 *
 * Provides support for managing selections across widgets (or a widget).
 *
 *)

signature SELECT_MGR =
  sig

    type select_mgr
    
    datatype select_widget = SelectWidget of {widget: Widget.widget, 
        selectfn: (Geometry.point * Geometry.point) -> (unit -> string),
        clearfn: unit -> unit}
    
    val mkSelectMgr : unit -> select_mgr
    
    val addWidget : select_mgr -> select_widget -> Widget.widget

  end (* SELECT_MGR *)

structure SelectMgr : SELECT_MGR = 
  struct

    structure W = Widget
    structure I = Interact
    structure EXB = EXeneBase
    
    datatype selectMsg = SelectBegin of (Geometry.point * EXB.XTime.time)
                       | SelectEnd   of (Geometry.point * EXB.XTime.time)
                       | SelectMove  of (Geometry.point * EXB.XTime.time)
                       | AddWidget   of (EXB.window * 
                            ((Geometry.point * Geometry.point) -> (unit -> string)) *
                            (unit -> unit))
    
    datatype select_mgr = 
        SelectMgr of {msgch: selectMsg CML.chan}
  
    datatype select_widget = 
        SelectWidget of {widget: W.widget, 
            selectfn: (Geometry.point * Geometry.point) -> (unit -> string),
            clearfn: unit -> unit}
    
    fun rectToStr (Geometry.RECT{x,y,wid,ht}) =
        ("RECT{x="^(Int.toString x)^",y="^(Int.toString y)^",wid="^
                            (Int.toString wid)^",ht="^(Int.toString ht)^"}")
    fun mkSelectMgr () =
        let
        val msgChan : selectMsg CML.chan = CML.channel()
        fun clearWidgets ([])          = ()
          | clearWidgets ((w,f,g)::l)  = (g(); clearWidgets(l))
        fun setWidgets (r as Geometry.RECT{x,y,wid,ht},[],sl)         = sl
          | setWidgets (r as Geometry.RECT{x,y,wid,ht},(w,f,g)::l,sl) = 
                let
                val {pos=Geometry.PT{x=wx,y=wy},sz=Geometry.SIZE{wid=ww,ht=wh},...} = 
                        (EXB.geomOfWin w)
                val Geometry.PT{x=wx,y=wy} = EXeneWin.winPtToScrPt w (Geometry.PT{x=0,y=0})
                val wr = Geometry.RECT{x=wx,y=wy,wid=ww,ht=wh}
                in 
                if (Geometry.intersect(r,wr))
                then let 
                     val i = Geometry.intersection(r,wr)
                     val Geometry.RECT{x=x1,y=y1,wid=iw,ht=ih} = i
                     val p1 = Geometry.PT{x=(x1-wx),y=(y1-wy)}
                     val p2 = Geometry.PT{x=(x1+iw),y=(y1+ih)}
                     val sf = f(p1,p2)  
                     in setWidgets(r,l,(w,sf,g)::sl) end
                else setWidgets(r,l,sl) end
        (** FIXME: must consolidate selections *)
        fun grabWidgets ([],t) = ()
          | grabWidgets ((w,f,g)::l,t) =
                 let
                 val str = f()
                 val sh = ICCC.acquireSelection (w,(ICCC.internAtom 
                            (EXeneWin.displayOfWin w) "PRIMARY"),t)
                 fun shLoop () =
                    case sh of
                        NONE => ()
                      | SOME sh => 
                    (let 
                     val reqEvt = ICCC.selectionReqEvt sh
                     val relEvt = ICCC.selectionRelEvt sh
                     in CML.select[
                            CML.wrap(reqEvt,fn {target,time,reply} => (
                                (** FIXME: allow multiple target types **
                                (TextIO.print ("[select-mgr: request for selection, target ="^
                                    (ICCC.nameOfAtom (EXeneWin.displayOfWin w) target)^"]\n");*)
                                (reply(SOME (ICCC.PROP_VAL{typ=ICCC.atom_STRING,
                                value=ICCC.RAW_DATA{format=ICCC.Raw8,
                                data=(Word8Vector.fromList (List.map
                                 (fn c => Word8.fromInt (Char.ord c)) 
                                 (String.explode str))) }}) ) );
                                shLoop())),
                            CML.wrap(relEvt,(fn () => (g())))
                        ]
                     end)
                 in (CML.spawn shLoop); () end                         
        fun ptsToRect (Geometry.PT{x=x1,y=y1},Geometry.PT{x=x2,y=y2}) =
            case ((x1<x2),(y1<y2)) of
                  (true,true)   => Geometry.RECT{x=x1,y=y1,wid=(x2-x1),ht=(y2-y1)}
                | (true,false)  => Geometry.RECT{x=x1,y=y2,wid=(x2-x1),ht=(y1-y2+1)}
                | (false,true)  => Geometry.RECT{x=x2,y=y1,wid=(x1-x2+1),ht=(y2-y1)}
                | (false,false) => Geometry.RECT{x=x2,y=y2,wid=(x1-x2+1),ht=(y1-y2+1)}
        
        fun loopStart (c,wl) = 
            (case (CML.recv msgChan) of
                SelectBegin(p1,t) => (clearWidgets(wl);loopFinish(c,wl,p1))
              | SelectMove(p2,t)  => loopStart(c,wl)
              | SelectEnd(p2,t)   => loopStart(c,wl)
              | AddWidget(w,f,g)  => loopStart((c+1),(w,f,g)::wl) )
        and loopFinish (c,wl,p1 as Geometry.PT{x=x1,y=y1}) = 
            (case (CML.recv msgChan) of
                SelectBegin(p1,t) => (clearWidgets(wl);loopFinish(c,wl,p1))
              | SelectMove(p2 as Geometry.PT{x=x2,y=y2},t) =>
                    let
                    val _ = setWidgets(ptsToRect(p1,p2),wl,[])
                    in (loopFinish(c,wl,p1)) end
              | SelectEnd(p2 as Geometry.PT{x=x2,y=y2},t) =>
                    let
                    val gl = setWidgets(ptsToRect(p1,p2),wl,[])
                    val _  = grabWidgets(gl,t)
                    in loopStart(c,wl) end
              | AddWidget(w,f,g)  => loopFinish((c+1),(w,f,g)::wl,p1) )
        in 
            (CML.spawn(fn () => (loopStart (0,[])) );
            SelectMgr{msgch=msgChan}) 
        end
        
    fun addWidget sm sw = 
        case (sm,sw) of (SelectMgr{msgch=mgrch},SelectWidget{widget=w,selectfn,clearfn}) =>
        let
        val mch : (I.mouse_msg I.addr_msg CML.chan) = CML.channel()
        fun realize2 {env=I.InEnv{k,m,ci,co},win,sz} = 
            let
            fun loop () = 
                (CML.select[
                    CML.wrap(m,fn m => 
                        ((case (I.msgBodyOf m) of
                            I.MOUSE_FirstDown{but=I.MButton(1),scr_pt,time,...} =>
                                CML.send(mgrch,SelectBegin(scr_pt,time)) 
                          | I.MOUSE_Motion{scr_pt,time,...} =>
                                CML.send(mgrch,SelectMove(scr_pt,time))
                          | I.MOUSE_LastUp{but=I.MButton(1),scr_pt,time,...} =>
                                CML.send(mgrch,SelectEnd(scr_pt,time))
                          | _ => ()
                         );
                         CML.send (mch,m)))
                ]; loop())
            in (CML.send(mgrch,AddWidget(win,selectfn,clearfn));CML.spawn(loop); 
                (W.realizeFn w) {env=I.InEnv{k=k,m=(CML.recvEvt mch),ci=ci,co=co},
                            win=win,sz=sz}) end
        in  (W.mkWidget{root=(W.rootOf w),args=(fn () => (W.argsOf w)),
                boundsOf=(fn () => (W.boundsOf w)),realize=realize2})
        end
    
    
end (* FocusMgr *)

