(* focus-mgr.sml
 *
 * (c) 2005 Dusty deBoer, Kansas State University, ddeboer@cis.ksu.edu
 *
 * Provides a widget for managing input focus among a set of widgets.
 *
 *)

signature FOCUS_MGR =
  sig

    type focus_mgr
    
    (* Widget will notify the manager when it has gained or lost focus. *)
    datatype notify_msg = FocusIn | FocusOut
    (* Widget will notify the manager when focus should be changed. *)
    datatype widget_msg = Assign | Release | Next | Previous
    datatype focusable  = 
        Focusable of {notifyEvt: notify_msg CML.event, 
                      widgetEvt: widget_msg CML.event,
                      takefocus: unit -> unit,
                      releasefocus: unit -> unit}
    
    val mkFocusMgr : unit -> focus_mgr
    
    val addFocusable : focus_mgr -> focusable -> unit
    
    val getFocusIndex : focus_mgr -> int option
    val setFocusIndex : focus_mgr -> int -> unit
    
  end (* FOCUS_MGR *)

structure FocusMgr : FOCUS_MGR = 
  struct

    structure W = Widget
    structure I = Interact

    datatype focusMsg = SetCurrent of  int 
                      | SetNext    of  int
                      | SetPrev    of  int
                      | AddWidget  of ((int SyncVar.ivar) * (unit->unit) * (unit->unit))
                      | GetCurrent of ((int option) SyncVar.ivar)
                      | UpdateCurrent of (int option)
    
    datatype focus_mgr = 
        FocusMgr of {msgch: focusMsg CML.chan}
  
    datatype notify_msg = FocusIn | FocusOut
    datatype widget_msg = Assign | Release | Next | Previous
    datatype focusable  = 
        Focusable of {notifyEvt: notify_msg CML.event, 
                      widgetEvt: widget_msg CML.event,
                      takefocus: unit -> unit,
                      releasefocus: unit -> unit}
    
    fun mkFocusMgr () =
        let
        val msgChan : focusMsg CML.chan = CML.channel()
        fun findf i []              = (fn () => ()) (* throw exception here? *)
          | findf i ((j,f,g)::l)      = (if i=j then f else findf i l)
        fun findg i []              = (fn () => ()) (* throw exception here? *)
          | findg i ((j,f,g)::l)      = (if i=j then g else findg i l)
        fun loop c wl (ci: int option) =
            (case (CML.recv msgChan) of
                SetCurrent(n)  => ((findf n wl)(); loop c wl ci)
              | SetNext(n)     => 
                    (let 
                     val i=(if (n+1)>=c then 0 else (n+1)) 
                     in (findf i wl)(); loop c wl ci end)
              | SetPrev(n)     => 
                    (let
                     val i=(if (n-1)<0 then (c-1) else (n-1))
                     in (findf i wl)(); loop c wl ci end)
              | AddWidget(iv,f,g) => (SyncVar.iPut(iv,c);(loop (c+1) ((c,f,g)::wl) ci) ) 
              | GetCurrent(iv)  => (SyncVar.iPut(iv,ci); loop c wl ci) 
              | UpdateCurrent(ci) => 
                   ((case ci of NONE => (* (findg 0 wl) *) () | SOME _ => ());
                    (loop c wl ci)) )
        in 
            (CML.spawn(fn () => (loop 0 [] NONE));
            FocusMgr{msgch=msgChan}) 
        end
        
    fun addFocusable fm fa = 
        case (fm,fa) of (FocusMgr{msgch=mgrch},
            Focusable{notifyEvt,widgetEvt,takefocus,releasefocus}) =>
        let
        val idv : int SyncVar.ivar = SyncVar.iVar()
        val _ = CML.send(mgrch,AddWidget(idv,takefocus,releasefocus))
        val id = SyncVar.iGet idv
        fun loop () = 
            (CML.select[
                CML.wrap(notifyEvt,fn n => 
                    (case n of
                        FocusIn  => (CML.send(mgrch,UpdateCurrent(SOME id)))
                      | FocusOut => (CML.send(mgrch,UpdateCurrent(NONE))) )),
                CML.wrap(widgetEvt,fn w => 
                    (case w of 
                        Assign   => (CML.send(mgrch,SetCurrent(id)))
                      | Release  => (CML.send(mgrch,UpdateCurrent(NONE)))
                      | Next     => (CML.send(mgrch,SetNext(id)))
                      | Previous => (CML.send(mgrch,SetPrev(id))) ))
            ]; loop())
        in (CML.spawn(loop); ()) end
    
    fun getFocusIndex (FocusMgr{msgch=mgrch}) =
        let
        val idv : int option SyncVar.ivar = SyncVar.iVar()
        val _ = CML.send(mgrch,GetCurrent(idv))
        in (SyncVar.iGet idv) end
    fun setFocusIndex (FocusMgr{msgch=mgrch}) i =
        CML.send(mgrch,SetCurrent(i))

end (* FocusMgr *)

