(* focus-frame.sml
 *
 * COPYRIGHT (c) 1994 by AT&T Bell Laboratories  See COPYRIGHT file for details.
 *
 * Frame widget, for putting a border around another widget
 * focus-frame.sml : ddeboer, KSU, handle border around focused widgets.
 *)

signature FOCUSFRAME = 
  sig

    structure W : WIDGET

    type frame

    val focusframe : (W.root * W.view * W.arg list) -> (W.widget * Shell.focusable) -> frame

    val widgetOf : frame -> W.widget
    
    val focusableOf : frame -> Shell.focusable

  end (* FOCUSFRAME *)

structure FocusFrame : FOCUSFRAME = 
  struct

    structure W = Widget
    structure D = Drawing
    structure I = Interact

    open CML Geometry

    val attrs = [
        ([], Attrs.attr_padx,           Attrs.AT_Int,      Attrs.AV_Int 0),
        ([], Attrs.attr_pady,           Attrs.AT_Int,      Attrs.AV_Int 0),
        ([], Attrs.attr_borderWidth,    Attrs.AT_Int,      Attrs.AV_Int 2),
        ([], Attrs.attr_relief,         Attrs.AT_Relief,   Attrs.AV_Relief (W.Sunken)),
        ([], Attrs.attr_background,     Attrs.AT_Color,    Attrs.AV_NoValue),
        ([], Attrs.attr_borderColor, Attrs.AT_Color, Attrs.AV_Str "black")
      ]

    type res = {
                 padx : int,
                 pady : int,
                 borderWidth : int,
                 relief : W.relief,
                 background : W.EXB.color option
               }

    datatype frame = Frame of {
        widget : W.widget,
        foc : Shell.focusable
      }

    fun mkResources (view,args) = let
          val attrs = W.findAttr (W.attrs (view,attrs,args))
          in {
               padx = Attrs.getInt (attrs Attrs.attr_padx),
               pady = Attrs.getInt (attrs Attrs.attr_pady),
               borderWidth = Attrs.getInt (attrs Attrs.attr_borderWidth),
               relief = Attrs.getRelief(attrs Attrs.attr_relief),
               background = Attrs.getColorOpt (attrs Attrs.attr_background)
             } end

    fun focusframe (root, view as (name,style), args) (widget,
            fa as Shell.Focusable{focusableEvt=cfevt,takefocus}) = let
          val res = mkResources (view,args)
          val realizeVar = SyncVar.iVar ()
          val fch = CML.channel()
          val view = (Styles.extendView (name,"focusframe"),style)
          val attrs = W.findAttr (W.attrs(view,attrs,args))
          val bordc = Attrs.getColor (attrs Attrs.attr_borderColor)
          val pen = D.newPen [D.PV_Foreground bordc, D.PV_LineWidth 2]
          fun fillfn (d,r,c) = let
                val p = D.newPen [D.PV_Foreground c]
                in D.fillRect d p r end
          fun fillborder (d,r,false) = ()
            | fillborder (d,r,true)  = (D.drawRect d pen r)

          fun size () = let
                fun incBase (W.DIM{base,incr,min,nat,max},extra) =
                  W.DIM{base=base+extra,incr=incr,min=min,nat=nat,max=max}
                val {x_dim, y_dim} = W.boundsOf widget
                val xextra = 2*(#padx res + #borderWidth res)
                val yextra = 2*(#pady res + #borderWidth res)
                in
                  {x_dim = incBase(x_dim,xextra), y_dim = incBase(y_dim,yextra)}
                end

          fun realizeFrame {env as I.InEnv{co=myco,...}, win, sz} = let
                val (my_inenv, my_outenv) = I.createWinEnv ()  
                val I.InEnv{ci=myci,...} = I.ignoreInput my_inenv

                fun childRect (SIZE{wid,ht}) = let
                      val xoff = #padx res + #borderWidth res
                      val yoff = #pady res + #borderWidth res
                      in
                        RECT{x = xoff,
                             y = yoff,
                             wid=Int.max(1,wid-(2*xoff)),
                             ht=Int.max(1,ht-(2*yoff))}
                      end

                val crect = childRect sz
                val cwin = W.wrapCreate(win, crect,W.argsOf widget)
                val (cinenv, coutenv) = I.createWinEnv ()
                val I.OutEnv{co=childco,...} = coutenv
                val drawable = D.drawableOfWin win

                fun main (rect, update, bord) = let
                    fun fill () = (D.clearDrawable drawable; fillborder(drawable,rect,bord))

                    fun handleCO (I.CO_ResizeReq r) = sync(myco (I.CO_ResizeReq r))
                      | handleCO (I.CO_KillReq) = W.EXW.destroyWin cwin

                    fun handleCI (I.CI_Resize (RECT{x,y,wid,ht})) =
                         (W.EXW.moveAndResizeWin cwin 
                           (childRect(SIZE{wid=wid,ht=ht}));
                         main(RECT{x=0,y=0,wid=wid,ht=ht},false,bord))
                      | handleCI (I.CI_Redraw _) = fill ()
                      | handleCI _ = ()

                    fun loop () =
                        (select [
                          wrap (myci,    loop o handleCI o I.msgBodyOf),
                          wrap (childco, loop o handleCO),
                          wrap (cfevt, fn m => 
                            case m of 
                              Shell.FocusIn  => 
                                 (CML.send(fch,Shell.FocusIn); 
                                    main (rect,true,true))
                            | Shell.FocusOut => 
                                 (CML.send(fch,Shell.FocusOut);
                                    main (rect,true,false))
                            | x =>   (CML.send(fch,x); loop()) )
                        ])
                    in
                        loop(if update then fill () else ())
                    end
                in
                    Router.routePair (env, my_outenv, coutenv);
                    W.realizeFn widget {
                      env = cinenv, 
                      win = cwin,
                      sz = sizeOfRect crect
                    };
                    W.EXW.mapWin cwin;
                    main (mkRect(originPt,sz),false,false)
                end

          fun initLoop () =
                select [
                  wrap(SyncVar.iGetEvt realizeVar, fn arg => realizeFrame arg),
                  wrap(cfevt, fn m => (CML.send(fch,m)))
                ]
          in
            spawn (fn () => initLoop ());
            Frame {
              widget=W.mkWidget {
                root=root,
                args= fn () => {background = NONE},
                boundsOf=size, 
                realize=fn arg => SyncVar.iPut(realizeVar,arg)
              },
              foc = Shell.Focusable{focusableEvt=(WidgetBase.wrapQueue (recvEvt fch)),
                takefocus=takefocus}
             }
          end

    fun widgetOf (Frame{widget,...}) = widget
    fun focusableOf (Frame{foc,...}) = foc
  end
