(* shell.sml
 *
 * COPYRIGHT (c) 1994 by AT&T Bell Laboratories  See COPYRIGHT file for details.
 *
 * Shell widget to provide interface between X library/window manager
 * and widgets.
 *
 * TODO: Allow mapping/unmapping of shells
 *       Cleanup and complete shell resource usage
 *)

signature SHELL = 
  sig

    structure W : WIDGET
    
    type shell

    (* type wm_args = { 
     *   win_name : string option, 
     *   icon_name : string option
     * }
     *)
    type wm_args
    val mkWMArgs : {win_name : string option, icon_name : string option} -> wm_args

    (* type hints = {
     *   size_hints : size_hints list,
     *   wm_hints : wm_hints list
     * }
     *)
    type hints
    val mkHints : {size_hints : W.EXW.ICCC.size_hints list, wm_hints : W.EXW.ICCC.wm_hints list } 
                    -> hints

    val shell : (W.root * W.view * W.arg list) -> W.widget -> shell
    
    val mkShell : W.widget * W.EXB.color option * wm_args -> shell
    val mkShellAt : W.G.rect -> W.widget * W.EXB.color option * wm_args -> shell
    
    val mkTransientShell : W.EXB.window -> 
          W.widget * W.EXB.color option * wm_args -> shell
    val mkTransientShellAt : W.G.rect -> W.EXB.window ->
          W.widget * W.EXB.color option * wm_args -> shell

    val setWMHints : shell -> hints -> unit
    val init : shell -> unit
    val map : shell -> unit
    val unmap : shell -> unit
    val destroy : shell -> unit
    
    (* added by ddeboer: *)
    datatype focusable_msg 
        = FocusIn
        | FocusOut 
        | Assign   of W.EXB.XTime.time
        | Release 
        | Next     of W.EXB.XTime.time
        | Previous of W.EXB.XTime.time
    datatype focusable = Focusable of {focusableEvt: focusable_msg CML.event,
                                       takefocus: W.EXB.XTime.time -> unit}
    type fid
    val addFocusableFirst  : shell -> focusable -> fid
    
    val addFocusableAfter  : shell -> fid * focusable -> fid
    val deleteFocusable    : shell -> fid -> unit
    
    val deletionEvent : shell -> unit CML.event
    
    (* widgets do not generally expose their windows to applications.
     * therefore, add a method for applications to request selections. *)
    val requestSelection : shell -> (ICCC.selection * ICCC.target * W.EXB.XTime.time)
            -> ICCC.value option
    (* and to set selections *)
    val acquireSelection : shell -> (ICCC.selection * W.EXB.XTime.time * ICCC.convertfn)
            -> ICCC.selection_handle option
    (* end added *)
    
  end (* SHELL *)

structure Shell : SHELL = 
  struct

    structure W = Widget
    structure ICCC = ICCC

    local open CML Geometry EXeneBase Interact EXeneWin Widget ICCC in

    type hints = {
      size_hints : size_hints list,
      wm_hints : wm_hints list
      (* class_hints : {res_class : string, res_name : string} option *)
    }
    fun mkHints a = a
    datatype shell_msg = Init | Destroy | Map of bool | Hints of hints
      (* added by ddeboer *) 
      | ReqSelection of (ICCC.selection * ICCC.target * W.EXB.XTime.time * 
                ICCC.value option SyncVar.ivar)
      | AcqSelection of (ICCC.selection * W.EXB.XTime.time * ICCC.convertfn *
                ICCC.selection_handle option SyncVar.ivar)

    (* modified by ddeboer; original:
    datatype shell = Shell of (shell_msg chan) *)
    datatype focusable_msg 
        = FocusIn
        | FocusOut 
        | Assign   of XTime.time
        | Release 
        | Next     of XTime.time
        | Previous of XTime.time
    datatype focusable = Focusable of {focusableEvt: focusable_msg CML.event,
                                       takefocus: XTime.time -> unit}
    datatype fidnode = 
        FID of {prev: fidnode ref, takefocus: XTime.time -> unit, next: fidnode ref, nbr: int}
      | NilFID
    type fid = fidnode ref
    datatype focus_msg = SetCurrent   of (fid * XTime.time)
                       | SetNext      of (fid * XTime.time)
                       | SetPrev      of (fid * XTime.time)
                       | AddFocusableFirst of ((fid SyncVar.ivar) * (XTime.time -> unit))
                       | AddFocusableAfter of (fid * (fid SyncVar.ivar) * (XTime.time -> unit))
                       | DeleteFocusable of fid
                       | GetCurrent   of ((fid option) SyncVar.ivar)
                       | UpdateCurrent of (fid option)
                       | ShellTakeFocus of XTime.time
                       | ShellFocusIn
                       | ShellFocusOut
    datatype shell = Shell of (shell_msg chan * unit chan * focus_msg chan)
    (* end modified *)
    
    fun setSizeHints {x_dim=x_dim as DIM xdim,y_dim=y_dim as DIM ydim} = let
          fun minSz () = let
                val minx = minDim x_dim
                val miny = minDim y_dim
                in
                  SIZE{wid=Int.max(1,minx),ht=Int.max(1,miny)}
                end
          fun maxSz () = (maxDim x_dim, maxDim y_dim)
          fun incSz () = (#incr xdim, #incr ydim)

          val MAX = 65535

          fun doInc () =
                case incSz () of
                  (1,1) => []
                | (x, 1) => [HINT_PResizeInc (SIZE{wid=x,ht=1})]
                | (1, y) => [HINT_PResizeInc (SIZE{wid=1,ht=y})]
                | (x, y) => [HINT_PResizeInc (SIZE{wid=x,ht=y})]

          fun doMin () = let
                val minsz = minSz ()
                in
                  [HINT_PMinSize minsz,HINT_PBaseSize minsz]
                end

          fun doMax () =
                case maxSz () of
                  (NONE,NONE) => []
                | (SOME x, NONE) => [HINT_PMaxSize (SIZE{wid=x,ht=MAX})]
                | (NONE, SOME y) => [HINT_PMaxSize (SIZE{wid=MAX,ht=y})]
                | (SOME x, SOME y) => [HINT_PMaxSize (SIZE{wid=x,ht=y})]
          in
            (doInc())@(doMax())@(doMin())
          end

(* DEBUG
    val setSizeHints = fn arg => let
          val pr = XDebug.pr1
          val arglist = setSizeHints arg
          fun pritem (HINT_PResizeInc sz) = pr("inc = "^(Db.sztos sz)^"\n")
            | pritem (HINT_PMaxSize sz) = pr("max = "^(Db.sztos sz)^"\n")
            | pritem (HINT_PMinSize sz) = pr("min = "^(Db.sztos sz)^"\n")
            | pritem _ = ()
          in
            app pritem arglist;
            arglist
          end
*)

    type wm_args = { win_name : string option, icon_name : string option }
    fun mkWMArgs a = a

    fun placement (NONE,sz) = (originPt,sz)
      | placement (SOME(RECT{x,y,wid,ht}),SIZE{wid=dfltwid,ht=dfltht}) =
          (PT{x=x,y=y},
           SIZE{wid= if wid > 0 then wid else dfltwid,
                ht= if ht > 0 then ht else dfltht})

    fun mk_shell crwin rectopt (widget, colorOpt, wm_args : wm_args) = let
          val root = rootOf widget
          val reqChan = channel ()
          val scr = screenOf root
          val color = 
            case colorOpt of 
              NONE => whiteOfScr scr
            | SOME color => color
      
          (* added by ddeboer: *)
          val delCh = CML.channel()
          val focCh = CML.channel()
          (* end added *)
      
          fun setProtocols win = 
                (* modified, ddeboer, to include WM_TAKE_FOCUS. *)
                setWMProtocols win [(ICCC.internAtom (displayOf root) "WM_DELETE_WINDOW"),
                                    (ICCC.internAtom (displayOf root) "WM_TAKE_FOCUS")]
            (* added ddeboer *)
            fun fidSerial (FID{nbr,...}) = Int.toString nbr
            fun handleFI (ctr,cur,lcur) (SetCurrent(fi,xt)) =
                    let
                    val FID{takefocus,...} = !fi
                    in (takefocus xt);(ctr,cur,lcur) end
              | handleFI (ctr,cur,lcur) (SetNext(fi,xt)) =
                    let
                    val FID{next,...} = !fi
                    val FID{takefocus,...} = !next
                    in (takefocus xt); (ctr,cur,lcur) end
              | handleFI (ctr,cur,lcur) (SetPrev(fi,xt)) =
                    let
                    val FID{prev,...} = !fi
                    val FID{takefocus,...} = !prev
                    in (takefocus xt); (ctr,cur,lcur) end
              | handleFI (ctr,cur,lcur) (AddFocusableFirst(iv,f)) =
                    let
                    val idn = FID{next=ref NilFID,prev=ref NilFID,takefocus=f,nbr=ctr}
                    val FID{next,prev,nbr,...} = idn
                    val _ = next:=idn
                    val _ = prev:=idn
                    val _ = SyncVar.iPut(iv,ref idn)
                    in (ctr+1,cur,lcur) end
              | handleFI (ctr,cur,lcur) (AddFocusableAfter(fid,iv,f)) =
                    let
                    val FID{next,...} = !fid
                    val FID{prev,...} = !next
                    val idn = FID{next=(ref (!next)),prev=(ref (!fid)),takefocus=f,nbr=ctr}
                    val _ = next:=idn
                    val _ = prev:=idn
                    val _ = SyncVar.iPut(iv,(ref idn))
                    in (ctr+1,cur,lcur) end
              | handleFI (ctr,cur,lcur) (DeleteFocusable(fid)) =
                    let
                    val FID{next,prev,...} = !fid
                    val FID{prev=np,...} = !next
                    val FID{next=pn,...} = !prev
                    val _ = np:=(!prev)
                    val _ = pn:=(!next)
                    in (ctr,cur,lcur) end
              | handleFI (ctr,cur,lcur) (GetCurrent(iv)) =
                    (SyncVar.iPut(iv,cur); (ctr,cur,lcur))
              | handleFI (ctr,cur,lcur) (UpdateCurrent(c)) =
                    (case c of NONE => (ctr,NONE,lcur)
                             | SOME n => (ctr,SOME n,SOME n))
              | handleFI (ctr,NONE,SOME fi) (ShellTakeFocus(xt)) = 
                    let
                    val FID{takefocus,...} = !fi
                    in (takefocus xt); (ctr,NONE,SOME fi) end
              | handleFI (ctr,cur,lcur) (ShellTakeFocus(xt)) = (ctr,cur,lcur)
              | handleFI (ctr,cur,lcur) (ShellFocusIn)   = (ctr,cur,lcur)
              | handleFI (ctr,cur,lcur) (ShellFocusOut)  = (ctr,cur,lcur)
            val focEvt = (WidgetBase.wrapQueue (recvEvt focCh))
            (* end added *)
                
          fun init (hintlist,mapped,focst) = let
                val bnds as {x_dim,y_dim} = boundsOf widget
                val dfltsize = SIZE{wid=natDim x_dim,ht=natDim y_dim}
                val (origin,size) = placement(rectopt,dfltsize)
                (* modified by ddeboer; original: 
                val (twin, inEnv) = crwin widget { ... *)
                val (twin, inEnv, inCMEChOpt) = crwin widget {
                  geom=WGEOM{pos=origin, sz=size, border=0},
                  backgrnd = color,
                  border = color   (* not used *)
                }

                fun sendHint {size_hints, wm_hints} =
                      setWMProperties twin {
                        argv = [],
                        win_name = NONE,
                        icon_name = NONE,
                        size_hints = size_hints,
                        wm_hints = wm_hints,
                        class_hints = NONE
                      }

                val _ = setWMProperties twin {
                          argv = SMLofNJ.getArgs(),
                          win_name = #win_name wm_args,
                          icon_name = #icon_name wm_args,
                          size_hints = setSizeHints bnds,
                          wm_hints = [],
                          class_hints = NONE
                        }
                val _ = app sendHint (rev hintlist)
                val _ = setProtocols twin

                val (my_inenv, my_outenv) = createWinEnv ()
    
                val cwin = wrapCreate(twin, mkRect(originPt,size),argsOf widget)
                val (cinenv, coutenv as OutEnv{co,...}) = createWinEnv ()
                val childco = wrapQueue co
                val InEnv{ci=myci,...} = ignoreInput my_inenv
    
                fun zombie () =
                      zombie (select [
                        wrap (myci, fn _ => ()),
                        wrap (recvEvt reqChan, fn _ => ()),
                        wrap (childco, fn _ => ())
                      ])
          
                fun handleCO (CO_ResizeReq r) = let (* < ddeboer: FIXME *)
                      val (bnds as {x_dim, y_dim}) = boundsOf widget
                      in
                  setWMProperties twin {
                    argv = [],
                    win_name = NONE,
                    icon_name = NONE,
                    size_hints = setSizeHints bnds,
                    wm_hints = [],
                    class_hints = NONE
                  };                
                  resizeWin twin (SIZE{wid=natDim x_dim,ht=natDim y_dim})
                      end
                  | handleCO CO_KillReq = (destroyWin twin; zombie())
          
                fun handleCI (CI_Resize (RECT{wid,ht,...})) = 
                      resizeWin cwin (SIZE{wid=wid,ht=ht})
                  | handleCI CI_OwnDeath = zombie ()
                  | handleCI (CI_ChildDeath _) = zombie ()
                  | handleCI (CI_Redraw _) = ()
                  | handleCI _ = ()
                
                fun mapTopWin (false,true) = (mapWin twin; true)
                  | mapTopWin (true, false) = (withdrawWin twin; false)
                  | mapTopWin (_,b) = b
                      
                fun handleReq mapped =
                      fn Init => mapped
                       | Destroy => (destroyWin twin; zombie ())
                       | Hints hint => (sendHint hint; mapped)
                       | Map arg => mapTopWin (mapped,arg)
                (* added by ddeboer *)
                       | ReqSelection (se,ta,ti,rv) =>
                          ((SyncVar.iPut(rv,CML.sync 
                            (ICCC.requestSelection (twin,se,ta,ti)) )); 
                          mapped)
                       | AcqSelection (se,ti,cf,rv) =>
                          ((SyncVar.iPut(rv,(ICCC.acquireSelection
                              (twin,se,ti,cf)))); mapped)
                (* modified by ddeboer: *)
                val inCMEvt = case inCMEChOpt of NONE => never 
                        | SOME inCMECh => WidgetBase.wrapQueue (recvEvt inCMECh)
                fun loop (mapped,f) =
                       (select [
                           wrap (myci, fn m => (case (msgBodyOf m) of 
                                CI_FocusIn  => (
                                    loop (mapped,(handleFI f ShellFocusIn)))
                              | CI_FocusOut => (
                                    loop (mapped,(handleFI f ShellFocusOut)))
                              | m1 => (
                                    handleCI m1)
                           )),
                           wrap (inCMEvt, 
                             fn xe => (case xe of
                                        CLIENT_TakeFocus(xt) => 
                                            (loop (mapped,(handleFI f (ShellTakeFocus xt))))
                                      | CLIENT_DeleteWindow(xt) => 
                                            (CML.send (delCh, ()) ) 
                                      )),
                           wrap (focEvt, (fn m => 
                                (loop (mapped,handleFI f m))) ),
                           wrap (recvEvt reqChan, (fn m => 
                                (loop ((handleReq mapped m),f))) ),
                           wrap (childco, handleCO)
                         ]; 
                       loop (mapped,f)) 
                in
                  Router.routePair (inEnv, my_outenv, coutenv);
                  realizeFn widget {
                    env = cinenv, 
                    win = cwin,
                    sz = size
                  };
                  mapWin cwin;
                  loop (mapTopWin(false, mapped), focst ) (* end modified *)
                end
  
          fun initLoop (arg as (hintlist,mapped,focst)) =
            (* modified ddeboer, was: 
            case recv reqChan of
              Init => init arg
            | Destroy => initLoop arg
            | Hints hint => initLoop (hint::hintlist,mapped)
            | Map mapped' => initLoop (hintlist,mapped') *)
            select [
                wrap(recvEvt reqChan, fn m => 
                        (case m of
                              Init => init arg
                            | Destroy => initLoop arg
                            | Hints hint => initLoop (hint::hintlist,mapped,focst)
                            | Map mapped' => initLoop (hintlist,mapped',focst)
                            | ReqSelection (se,ta,ti,rv) =>
                                (SyncVar.iPut(rv,NONE); initLoop arg)
                            | AcqSelection (se,ti,cf,rv) =>
                                (SyncVar.iPut(rv,NONE); initLoop arg)
                        )),
                wrap(focEvt, fn m => (initLoop (hintlist,mapped,(handleFI focst m))))
            ]
  
          in
            XDebug.xspawn ("shell", fn () => initLoop ([],true,(0,NONE,NONE)));
            (* modified by ddeboer; original:
            Shell reqChan *) 
            (Shell (reqChan, delCh, focCh))
          end
  
    local
      (* modified by ddeboer; original: 
      fun simple wdgt = createSimpleTopWin (screenOf(rootOf wdgt)) 
      fun trans w _ = createTransientWin w*)
      fun simple wdgt g = 
        let
        val (win,inEnv,cmeCh) = (createSimpleTopWin (screenOf(rootOf wdgt)) g)
        in (win,inEnv,SOME cmeCh) end
      fun trans w _ g = 
        let
        val (win,inEnv) = createTransientWin w g
        in (win,inEnv,NONE) end (* end modified *)
    in
    fun mkShellAt r = mk_shell simple (SOME r)
    val mkShell = mk_shell simple NONE
    fun mkTransientShellAt r w = mk_shell (trans w) (SOME r)
    fun mkTransientShell w = mk_shell (trans w) NONE
    val attrs = [
        ([], Attrs.attr_title,          Attrs.AT_Str,    Attrs.AV_NoValue),
        ([], Attrs.attr_iconName,       Attrs.AT_Str,    Attrs.AV_NoValue),
        ([], Attrs.attr_background,     Attrs.AT_Color,  Attrs.AV_NoValue)
      ]

    fun shell (root, view ,args) widget = let
          val attrs = W.findAttr (W.attrs (view, attrs,args))
          val win_name = Attrs.getStringOpt (attrs Attrs.attr_title)
          val icon_name = Attrs.getStringOpt (attrs Attrs.attr_iconName)
          val pos = NONE (* FIX to lookup geometry *)
          val color = Attrs.getColorOpt (attrs Attrs.attr_background)
          val args = {win_name = win_name, icon_name = icon_name}
          in mk_shell simple pos (widget, color, args) end 

    end (* local *)

    (* following modified by ddeboer; original:
    fun init (Shell ch) = send (ch, Init)
    fun destroy (Shell ch) = send (ch, Destroy)
    fun unmap (Shell ch) = send(ch, Map false)
    fun map (Shell ch) = send(ch, Map true)
    fun setWMHints (Shell ch) arg = send (ch, Hints arg) *)
    fun init (Shell (ch,cmech,fch)) = send (ch, Init)
    fun destroy (Shell (ch,cmech,fch)) = send (ch, Destroy)
    fun unmap (Shell (ch,cmech,fch)) = send(ch, Map false)
    fun map (Shell (ch,cmech,fch)) = send(ch, Map true)
    fun setWMHints (Shell (ch,cmech,fch)) arg = send (ch, Hints arg)
    fun deletionEvent (Shell (ch,cmech,fch)) = (recvEvt cmech)
    (* end modified *)
    
    (* following added by ddeboer *)
    fun requestSelection (Shell (ch,cmech,fch)) (se,ta,ti) = 
            let
            val repv : ICCC.value option SyncVar.ivar = SyncVar.iVar()
            in send (ch,ReqSelection (se,ta,ti,repv)); SyncVar.iGet repv end
    fun acquireSelection (Shell (ch,cmech,fch)) (se,ti,cf) = 
            let
            val repv : ICCC.selection_handle option SyncVar.ivar = SyncVar.iVar()
            in send (ch,AcqSelection (se,ti,cf,repv)); SyncVar.iGet repv end
    fun spawnLoop (id,mgrch,focusableEvt) =
        let
        fun loop () = 
            ((case (CML.sync focusableEvt) of
                    FocusIn      => (CML.send(mgrch,UpdateCurrent(SOME id)))
                  | FocusOut     => (CML.send(mgrch,UpdateCurrent(NONE)))
                  | Assign(xt)   => (CML.send(mgrch,SetCurrent(id,xt)))
                  | Release      => (CML.send(mgrch,UpdateCurrent(NONE)))
                  | Next(xt)     => (CML.send(mgrch,SetNext(id,xt)))
                  | Previous(xt) => (CML.send(mgrch,SetPrev(id,xt))) ); 
            loop())
        in (CML.spawn(loop); (id)) end  
        
    fun addFocusableFirst (Shell(ch,chech,mgrch)) 
            (Focusable{focusableEvt,takefocus}) = 
        let
        val idv : fid SyncVar.ivar = SyncVar.iVar()
        val _ = CML.send(mgrch,AddFocusableFirst(idv,takefocus))
        val id = SyncVar.iGet idv
        in spawnLoop (id,mgrch,focusableEvt) end
        
    fun addFocusableAfter (Shell(ch,chech,mgrch)) 
            (fid,Focusable{focusableEvt,takefocus}) = 
        let
        val idv : fid SyncVar.ivar = SyncVar.iVar()
        val _ = CML.send(mgrch,AddFocusableAfter(fid,idv,takefocus))
        val id = SyncVar.iGet idv
        in spawnLoop (id,mgrch,focusableEvt) end
        
    fun deleteFocusable (Shell(ch,chech,mgrch)) (fid) = 
        (CML.send(mgrch,DeleteFocusable(fid)))
        
    end (* local *)
  end (* Shell *)
