(* iccc-selection.sml
 *
 * @author ddeboer
 *)

structure ICCCSelection : sig

    datatype selection = Sel_PRIMARY
                       | Sel_SECONDARY
                       | Sel_CLIPBOARD
                       | Sel_OTHER of XProtTypes.atom
                       
    datatype target = Tgt_TARGETS
                    | Tgt_MULTIPLE of (target list)
                    | Tgt_TIMESTAMP
                    | Tgt_STRING
                    | Tgt_COMPOUND_TEXT
                    | Tgt_TEXT
                    | Tgt_LIST_LENGTH
                    | Tgt_PIXMAP
                    | Tgt_DRAWABLE
                    | Tgt_BITMAP
                    | Tgt_FOREGROUND
                    | Tgt_BACKGROUND
                    | Tgt_COLORMAP
                    | Tgt_ODIF
                    | Tgt_OWNER_OS
                    | Tgt_FILE_NAME
                    | Tgt_HOST_NAME
                    | Tgt_CHARACTER_POSITION
                    | Tgt_LINE_NUMBER
                    | Tgt_COLUMN_NUMBER
                    | Tgt_LENGTH
                    | Tgt_USER
                    | Tgt_PROCEDURE
                    | Tgt_MODULE
                    | Tgt_PROCESS
                    | Tgt_TASK
                    | Tgt_CLASS
                    | Tgt_NAME
                    | Tgt_CLIENT_WINDOW
                    | Tgt_DELETE
                    | Tgt_INSERT_SELECTION
                    | Tgt_INSERT_PROPERTY
                    | Tgt_UNKNOWN of string
    
    val tgtToString : target -> string
    
    datatype value  = Val_TARGETS           of target list
                    | Val_MULTIPLE          of value list
                    | Val_TIMESTAMP         of XTime.time
                    | Val_STRING            of string   
                    | Val_COMPOUND_TEXT     of string
                    | Val_LIST_LENGTH       of int
                    | Val_PIXMAP            of XProtTypes.pixmap_id list
                    | Val_DRAWABLE          of XProtTypes.drawable_id list
                    | Val_BITMAP            of XProtTypes.pixmap_id list
                    | Val_FOREGROUND        of XProtTypes.xid (* << pixel *)
                    | Val_BACKGROUND        of XProtTypes.xid (* << pixel *)
                    | Val_COLORMAP          of XProtTypes.colormap_id list
                    | Val_ODIF              of string
                    | Val_OWNER_OS          of string
                    | Val_FILE_NAME         of string
                    | Val_HOST_NAME         of string
                    | Val_CHARACTER_POSITION of (int * int)
                    | Val_LINE_NUMBER       of (int * int)
                    | Val_COLUMN_NUMBER     of (int * int)
                    | Val_LENGTH            of int
                    | Val_USER              of string
                    | Val_PROCEDURE         of string
                    | Val_MODULE            of string
                    | Val_PROCESS           of int
                    | Val_TASK              of int
                    | Val_CLASS             of string
                    | Val_NAME              of string
                    | Val_CLIENT_WINDOW     of XProtTypes.win_id
                    | Val_DELETE            
                    | Val_INSERT_SELECTION
                    | Val_INSERT_PROPERTY 
    
    (* selection aquirer must provide function capable of converting value to target. *)
    type convertfn = target -> value option
    (* utility function for hiding the conversion of string value to targets
     * that a string value can reasonably be converted to. *)
    val convertString : string -> target -> value option
    
    type selection_handle
    
    val releaseEvt : selection_handle -> unit CML.event
    val releaseSelection : selection_handle -> unit
    
    (* Hide the handling of TARGETS, MULTIPLE, and TIMESTAMP targets,
     * as well as handling of incremental transfers. 
     * Return true iff the acquiring window owns the selection. *)
    val acquireSelection : (Window.window * selection * XTime.time * convertfn) 
            -> selection_handle option
    
    (* hide the allocation/deallocation of a property for transfer of selection. *)
    val requestSelection : (Window.window * selection * target * XTime.time) 
            -> value option CML.event
    val requestSelectionString : (Window.window * selection * XTime.time)
            -> string option CML.event

    val storeCutBuffer : (Window.window * string) -> unit 
    val fetchCutBuffer : (Window.window) -> string option
    val rotateCutBuffer : (Window.window * int) -> unit
    
  end = struct
    
    datatype selection = Sel_PRIMARY
                       | Sel_SECONDARY
                       | Sel_CLIPBOARD
                       | Sel_OTHER of XProtTypes.atom
                       
    fun selToAtom dpy (Sel_PRIMARY)   = StdAtoms.atom_PRIMARY
      | selToAtom dpy (Sel_SECONDARY) = StdAtoms.atom_SECONDARY
      | selToAtom dpy (Sel_CLIPBOARD) = XAtoms.internAtom dpy "CLIPBOARD"
      | selToAtom dpy (Sel_OTHER(a))  = a
                       
    datatype target = Tgt_TARGETS
                    | Tgt_MULTIPLE of (target list)
                    | Tgt_TIMESTAMP
                    | Tgt_STRING
                    | Tgt_COMPOUND_TEXT
                    | Tgt_TEXT
                    | Tgt_LIST_LENGTH
                    | Tgt_PIXMAP
                    | Tgt_DRAWABLE
                    | Tgt_BITMAP
                    | Tgt_FOREGROUND
                    | Tgt_BACKGROUND
                    | Tgt_COLORMAP
                    | Tgt_ODIF
                    | Tgt_OWNER_OS
                    | Tgt_FILE_NAME
                    | Tgt_HOST_NAME
                    | Tgt_CHARACTER_POSITION
                    | Tgt_LINE_NUMBER
                    | Tgt_COLUMN_NUMBER
                    | Tgt_LENGTH
                    | Tgt_USER
                    | Tgt_PROCEDURE
                    | Tgt_MODULE
                    | Tgt_PROCESS
                    | Tgt_TASK
                    | Tgt_CLASS
                    | Tgt_NAME
                    | Tgt_CLIENT_WINDOW
                    | Tgt_DELETE
                    | Tgt_INSERT_SELECTION
                    | Tgt_INSERT_PROPERTY
                    | Tgt_UNKNOWN of string
                    
    datatype value  = Val_TARGETS           of target list
                    | Val_MULTIPLE          of value list
                    | Val_TIMESTAMP         of XTime.time
                    | Val_STRING            of string   
                    | Val_COMPOUND_TEXT     of string
                    (* | Val_TEXT              of string ** Not a valid value.*)
                    | Val_LIST_LENGTH       of int
                    | Val_PIXMAP            of XProtTypes.pixmap_id list
                    | Val_DRAWABLE          of XProtTypes.drawable_id list
                    | Val_BITMAP            of XProtTypes.pixmap_id list
                    | Val_FOREGROUND        of XProtTypes.xid (* << pixel *)
                    | Val_BACKGROUND        of XProtTypes.xid (* << pixel *)
                    | Val_COLORMAP          of XProtTypes.colormap_id list
                    | Val_ODIF              of string
                    | Val_OWNER_OS          of string
                    | Val_FILE_NAME         of string
                    | Val_HOST_NAME         of string
                    | Val_CHARACTER_POSITION of (int * int)
                    | Val_LINE_NUMBER       of (int * int)
                    | Val_COLUMN_NUMBER     of (int * int)
                    | Val_LENGTH            of int
                    | Val_USER              of string
                    | Val_PROCEDURE         of string
                    | Val_MODULE            of string
                    | Val_PROCESS           of int
                    | Val_TASK              of int
                    | Val_CLASS             of string
                    | Val_NAME              of string
                    | Val_CLIENT_WINDOW     of XProtTypes.win_id
                    | Val_DELETE            
                    | Val_INSERT_SELECTION
                    | Val_INSERT_PROPERTY 
    
    (* returns the number of bytes required to represent the value in a property. *)
    fun sizeOfVal (Val_TARGETS(l))              = (4*(List.length l))
      | sizeOfVal (Val_MULTIPLE(l))             = 
            (List.foldl (fn (v,a) => (a+(sizeOfVal v))) 0 l)
      | sizeOfVal (Val_TIMESTAMP(_))            = 4
      | sizeOfVal (Val_STRING(s))               = (String.size s)
      | sizeOfVal (Val_COMPOUND_TEXT(s))        = (String.size s)
      | sizeOfVal (Val_LIST_LENGTH(_))          = 4
      | sizeOfVal (Val_PIXMAP(l))               = (4*(List.length l))
      | sizeOfVal (Val_DRAWABLE(l))             = (4*(List.length l))
      | sizeOfVal (Val_BITMAP(l))               = (4*(List.length l))
      | sizeOfVal (Val_FOREGROUND(_))           = 4
      | sizeOfVal (Val_BACKGROUND(_))           = 4
      | sizeOfVal (Val_COLORMAP(l))             = (4*(List.length l))
      | sizeOfVal (Val_ODIF(s))                 = (String.size s)
      | sizeOfVal (Val_OWNER_OS(s))             = (String.size s)
      | sizeOfVal (Val_FILE_NAME(s))            = (String.size s)
      | sizeOfVal (Val_HOST_NAME(s))            = (String.size s)
      | sizeOfVal (Val_CHARACTER_POSITION(_))   = 8
      | sizeOfVal (Val_LINE_NUMBER(_))          = 8
      | sizeOfVal (Val_COLUMN_NUMBER(_))        = 8
      | sizeOfVal (Val_LENGTH(_))               = 4
      | sizeOfVal (Val_USER(s))                 = (String.size s)
      | sizeOfVal (Val_PROCEDURE(s))            = (String.size s)
      | sizeOfVal (Val_MODULE(s))               = (String.size s)
      | sizeOfVal (Val_PROCESS(_))              = 4
      | sizeOfVal (Val_TASK(_))                 = 4
      | sizeOfVal (Val_CLASS(s))                = (String.size s)
      | sizeOfVal (Val_NAME(s))                 = (String.size s)
      | sizeOfVal (Val_CLIENT_WINDOW(_))        = 4
      | sizeOfVal (Val_DELETE)                  = 0 (* fixme *)
      | sizeOfVal (Val_INSERT_SELECTION)        = 0 (* fixme *)
      | sizeOfVal (Val_INSERT_PROPERTY)         = 0 (* fixme *)
    
    fun tgtOfVal (Val_TARGETS(_))              = Tgt_TARGETS
      | tgtOfVal (Val_MULTIPLE(_))             = Tgt_MULTIPLE []
      | tgtOfVal (Val_TIMESTAMP(_))            = Tgt_TIMESTAMP
      | tgtOfVal (Val_STRING(_))               = Tgt_STRING
      | tgtOfVal (Val_COMPOUND_TEXT(_))        = Tgt_COMPOUND_TEXT
      | tgtOfVal (Val_LIST_LENGTH(_))          = Tgt_LIST_LENGTH
      | tgtOfVal (Val_PIXMAP(_))               = Tgt_PIXMAP
      | tgtOfVal (Val_DRAWABLE(_))             = Tgt_DRAWABLE
      | tgtOfVal (Val_BITMAP(_))               = Tgt_BITMAP
      | tgtOfVal (Val_FOREGROUND(_))           = Tgt_FOREGROUND
      | tgtOfVal (Val_BACKGROUND(_))           = Tgt_BACKGROUND
      | tgtOfVal (Val_COLORMAP(_))             = Tgt_COLORMAP
      | tgtOfVal (Val_ODIF(_))                 = Tgt_ODIF
      | tgtOfVal (Val_OWNER_OS(_))             = Tgt_OWNER_OS
      | tgtOfVal (Val_FILE_NAME(_))            = Tgt_FILE_NAME
      | tgtOfVal (Val_HOST_NAME(_))            = Tgt_HOST_NAME
      | tgtOfVal (Val_CHARACTER_POSITION(_))   = Tgt_CHARACTER_POSITION
      | tgtOfVal (Val_LINE_NUMBER(_))          = Tgt_LINE_NUMBER
      | tgtOfVal (Val_COLUMN_NUMBER(_))        = Tgt_COLUMN_NUMBER
      | tgtOfVal (Val_LENGTH(_))               = Tgt_LENGTH
      | tgtOfVal (Val_USER(_))                 = Tgt_USER
      | tgtOfVal (Val_PROCEDURE(_))            = Tgt_PROCEDURE
      | tgtOfVal (Val_MODULE(_))               = Tgt_MODULE
      | tgtOfVal (Val_PROCESS(_))              = Tgt_PROCESS
      | tgtOfVal (Val_TASK(_))                 = Tgt_TASK
      | tgtOfVal (Val_CLASS(_))                = Tgt_CLASS
      | tgtOfVal (Val_NAME(_))                 = Tgt_NAME
      | tgtOfVal (Val_CLIENT_WINDOW(_))        = Tgt_CLIENT_WINDOW
      | tgtOfVal (Val_DELETE)                  = Tgt_DELETE
      | tgtOfVal (Val_INSERT_SELECTION)        = Tgt_INSERT_SELECTION
      | tgtOfVal (Val_INSERT_PROPERTY)         = Tgt_INSERT_PROPERTY
      
    fun tgtToString (Tgt_TARGETS)            = "TARGETS"
      | tgtToString (Tgt_MULTIPLE _)         = "MULTIPLE"
      | tgtToString (Tgt_TIMESTAMP)          = "TIMESTAMP"
      | tgtToString (Tgt_STRING)             = "STRING"
      | tgtToString (Tgt_COMPOUND_TEXT)      = "COMPOUND_TEXT"
      | tgtToString (Tgt_TEXT)               = "TEXT"
      | tgtToString (Tgt_LIST_LENGTH)        = "LIST_LENGTH"
      | tgtToString (Tgt_PIXMAP)             = "PIXMAP"
      | tgtToString (Tgt_DRAWABLE)           = "DRAWABLE"
      | tgtToString (Tgt_BITMAP)             = "BITMAP"
      | tgtToString (Tgt_FOREGROUND)         = "FOREGROUND"
      | tgtToString (Tgt_BACKGROUND)         = "BACKGROUND"
      | tgtToString (Tgt_COLORMAP)           = "COLORMAP"
      | tgtToString (Tgt_ODIF)               = "ODIF"
      | tgtToString (Tgt_OWNER_OS)           = "OWNER_OS"
      | tgtToString (Tgt_FILE_NAME)          = "FILE_NAME"
      | tgtToString (Tgt_HOST_NAME)          = "HOST_NAME"
      | tgtToString (Tgt_CHARACTER_POSITION) = "CHARACTER_POSITION"
      | tgtToString (Tgt_LINE_NUMBER)        = "LINE_NUMBER"
      | tgtToString (Tgt_COLUMN_NUMBER)      = "COLUMN_NUMBER"
      | tgtToString (Tgt_LENGTH)             = "LENGTH"
      | tgtToString (Tgt_USER)               = "USER"
      | tgtToString (Tgt_PROCEDURE)          = "PROCEDURE"
      | tgtToString (Tgt_MODULE)             = "MODULE"
      | tgtToString (Tgt_PROCESS)            = "PROCESS"
      | tgtToString (Tgt_TASK)               = "TASK"
      | tgtToString (Tgt_CLASS)              = "CLASS"
      | tgtToString (Tgt_NAME)               = "NAME"
      | tgtToString (Tgt_CLIENT_WINDOW)      = "CLIENT_WINDOW"
      | tgtToString (Tgt_DELETE)             = "DELETE"
      | tgtToString (Tgt_INSERT_SELECTION)   = "INSERT_SELECTION"
      | tgtToString (Tgt_INSERT_PROPERTY)    = "INSERT_PROPERTY" 
      | tgtToString (Tgt_UNKNOWN s)          = s
    fun tgtToAtom dpy tgt = XAtoms.internAtom dpy (tgtToString tgt)
    
    fun atomToTgt dpy atm =
        (case (XAtoms.nameOfAtom dpy atm) of
             "TARGETS"               => (Tgt_TARGETS)             
           | "MULTIPLE"              => (Tgt_MULTIPLE [])            
           | "TIMESTAMP"             => (Tgt_TIMESTAMP)           
           | "STRING"                => (Tgt_STRING)             
           | "COMPOUND_TEXT"         => (Tgt_COMPOUND_TEXT)       
           | "TEXT"                  => (Tgt_TEXT)                
           | "LIST_LENGTH"           => (Tgt_LIST_LENGTH)         
           | "PIXMAP"                => (Tgt_PIXMAP)              
           | "DRAWABLE"              => (Tgt_DRAWABLE)            
           | "BITMAP"                => (Tgt_BITMAP)              
           | "FOREGROUND"            => (Tgt_FOREGROUND)          
           | "BACKGROUND"            => (Tgt_BACKGROUND)          
           | "COLORMAP"              => (Tgt_COLORMAP)            
           | "ODIF"                  => (Tgt_ODIF)                
           | "OWNER_OS"              => (Tgt_OWNER_OS)            
           | "FILE_NAME"             => (Tgt_FILE_NAME)           
           | "HOST_NAME"             => (Tgt_HOST_NAME)           
           | "CHARACTER_POSITION"    => (Tgt_CHARACTER_POSITION)  
           | "LINE_NUMBER"           => (Tgt_LINE_NUMBER)         
           | "COLUMN_NUMBER"         => (Tgt_COLUMN_NUMBER)       
           | "LENGTH"                => (Tgt_LENGTH)              
           | "USER"                  => (Tgt_USER)                
           | "PROCEDURE"             => (Tgt_PROCEDURE)           
           | "MODULE"                => (Tgt_MODULE)              
           | "PROCESS"               => (Tgt_PROCESS)             
           | "TASK"                  => (Tgt_TASK)                
           | "CLASS"                 => (Tgt_CLASS)               
           | "NAME"                  => (Tgt_NAME)                
           | "CLIENT_WINDOW"         => (Tgt_CLIENT_WINDOW)       
           | "DELETE"                => (Tgt_DELETE)              
           | "INSERT_SELECTION"      => (Tgt_INSERT_SELECTION)    
           | "INSERT_PROPERTY"       => (Tgt_INSERT_PROPERTY)  
           | na                      => (Tgt_UNKNOWN na))
      
    fun equalTgtType dpy (tgt1,tgt2) = ((tgtToAtom dpy tgt1)=(tgtToAtom dpy tgt2))
    fun atomOfVal dpy val1 = (tgtToAtom dpy (tgtOfVal val1))
    fun equalValType dpy (val1,val2) = ((atomOfVal dpy val1)=(atomOfVal dpy val2)) 
    
    val v2a : Word8Vector.vector -> Word8Array.array = Unsafe.cast
    fun strToPropVal (t,s) = (XProtTypes.PROP_VAL{typ=t,
             value=XProtTypes.RAW_DATA{format=XProtTypes.Raw8,data=(Byte.stringToBytes s)}})
    fun wordListToPropVal (t,wl) = (XProtTypes.PROP_VAL{typ=t,
             value=XProtTypes.RAW_DATA{format=XProtTypes.Raw32,
                data=let 
                val mask32 = Word32.fromLargeInt 0xff000000
                val mask24 = Word32.fromLargeInt 0xff0000
                val mask16 = Word32.fromLargeInt 0xff00
                val mask8  = Word32.fromLargeInt 0xff
                fun w2w8l w = [
                    (Word8.fromLargeWord (LargeWord.>> ((LargeWord.andb (mask32,w)),0w24))),
                    (Word8.fromLargeWord (LargeWord.>> ((LargeWord.andb (mask24,w)),0w16))),
                    (Word8.fromLargeWord (LargeWord.>> ((LargeWord.andb (mask16,w)),0w8))),
                    (Word8.fromLargeWord (LargeWord.andb (mask8,w)) )]
                fun wl2w8l [] = []
                  | wl2w8l (w::ws) = (w2w8l w)@(wl2w8l ws)
                in (Word8Vector.fromList (wl2w8l wl)) end}})
    fun atomListToPropVal (t,al) = wordListToPropVal (t,
        (List.map (fn (XProtTypes.XAtom w) => (Word.toLargeWord w)) al))
    fun idListToPropVal (t,il) = wordListToPropVal (t,
        (List.map (fn (XProtTypes.XID w) => (Word.toLargeWord w)) il))
    fun intListToPropVal (t,il) = wordListToPropVal (t,
        (List.map (fn i => (LargeWord.fromLargeInt i)) il))
                
    val fixme = XProtTypes.PROP_VAL{typ=StdAtoms.atom_STRING,value=XProtTypes.RAW_DATA{
            format=XProtTypes.Raw8,data=(Byte.stringToBytes "FIXME")}}
        
    fun valToPropVal dpy NONE = NONE
      | valToPropVal dpy (SOME val1) =
        let
        fun nullPropVal () = XProtTypes.PROP_VAL{typ=(XAtoms.internAtom dpy "NULL"),
                value=XProtTypes.RAW_DATA{format=XProtTypes.Raw8,data=(Word8Vector.fromList [])}}
        val ptype = XAtoms.internAtom dpy
        in case (val1) of
            (Val_TARGETS(l))              => 
                SOME (atomListToPropVal (ptype "ATOM",
                    List.map (fn t => tgtToAtom dpy t) l))
          | (Val_MULTIPLE(_))             => SOME (fixme)
          | (Val_TIMESTAMP(XTime.XT w))   => 
                SOME (intListToPropVal (ptype "INTEGER",
                [(Word32.toLargeInt w)]))
          | (Val_STRING(s))               => 
                SOME (strToPropVal (ptype "STRING",s))
          | (Val_COMPOUND_TEXT(s))        => 
                SOME (strToPropVal (ptype "COMPOUND_TEXT",s))
          | (Val_LIST_LENGTH(i))          => 
                SOME (intListToPropVal (ptype "INTEGER",[(Int.toLarge i)]))
          | (Val_PIXMAP(l))               => 
                SOME (idListToPropVal (ptype "DRAWABLE",l))
          | (Val_DRAWABLE(l))             => 
                SOME (idListToPropVal (ptype "DRAWABLE",l))
          | (Val_BITMAP(l))               => 
                SOME (idListToPropVal (ptype "BITMAP",l))
          | (Val_FOREGROUND(p))           => 
                SOME (idListToPropVal (ptype "PIXEL",[p]))
          | (Val_BACKGROUND(p))           => 
                SOME (idListToPropVal (ptype "PIXEL",[p]))
          | (Val_COLORMAP(l))             => 
                SOME (idListToPropVal (ptype "COLORMAP",l))
          | (Val_ODIF(s))                 => 
                SOME (strToPropVal (ptype "STRING",s))
          | (Val_OWNER_OS(s))             => 
                SOME (strToPropVal (ptype "STRING",s))
          | (Val_FILE_NAME(s))            => 
                SOME (strToPropVal (ptype "STRING",s))
          | (Val_HOST_NAME(s))            => 
                SOME (strToPropVal (ptype "STRING",s))
          | (Val_CHARACTER_POSITION(i,j)) => 
                SOME (intListToPropVal (ptype "SPAN",[(Int.toLarge i),(Int.toLarge j)]))
          | (Val_LINE_NUMBER(i,j))        => 
                SOME (intListToPropVal (ptype "SPAN",[(Int.toLarge i),(Int.toLarge j)]))
          | (Val_COLUMN_NUMBER(i,j))      => 
                SOME (intListToPropVal (ptype "SPAN",[(Int.toLarge i),(Int.toLarge j)]))
          | (Val_LENGTH(i))               => 
                SOME (intListToPropVal (ptype "INTEGER",[(Int.toLarge i)]))
          | (Val_USER(s))                 => 
                SOME (strToPropVal (ptype "STRING",s))
          | (Val_PROCEDURE(s))            => 
                SOME (strToPropVal (ptype "STRING",s))
          | (Val_MODULE(s))               => 
                SOME (strToPropVal (ptype "STRING",s))
          | (Val_PROCESS(i))              => 
                SOME (intListToPropVal (ptype "INTEGER",[(Int.toLarge i)]))
          | (Val_TASK(i))                 => 
                SOME (intListToPropVal (ptype "INTEGER",[(Int.toLarge i)]))
          | (Val_CLASS(s))                => 
                SOME (strToPropVal (ptype "STRING",s))
          | (Val_NAME(s))                 => 
                SOME (strToPropVal (ptype "STRING",s))
          | (Val_CLIENT_WINDOW(w))        => 
                SOME (idListToPropVal (ptype "WINDOW",[w]))
          | (Val_DELETE)                  => 
                SOME (nullPropVal ()) (* side effect *)
          | (Val_INSERT_SELECTION)        => 
                SOME (nullPropVal ()) (* side effect *)
          | (Val_INSERT_PROPERTY)         => 
                SOME (nullPropVal ()) (* side effect *)
       end
    
    fun mkIncr dpy lb = (XProtTypes.PROP_VAL{typ=(XAtoms.internAtom dpy "INCR"),
                 value=XProtTypes.RAW_DATA{format=XProtTypes.Raw32,
                    data=let 
                    val v2a : Word8Vector.vector -> Word8Array.array = Unsafe.cast
                    val buf = Unsafe.Word8Vector.create 4 
                    val _   = Pack32Big.update (v2a buf,0,(LargeWord.fromInt lb))
                    in buf end}})
                    
    (* functions to convert property values into other types *)
    fun valToString (XProtTypes.RAW_DATA{format=XProtTypes.Raw8,data}) = 
                (Byte.bytesToString data)
      | valToString (_) = ""
    fun getWord16 n v = 
        let
        val w1 = LargeWord.<< ((Word8.toLargeWord (Word8Vector.sub (v,(n*2)))),0w8)
        val w2 = (Word8.toLargeWord (Word8Vector.sub (v,(n*2)+1)))
        val w = Word.fromLargeWord (LargeWord.orb (w1,w2))
        in w end
    fun getWord32 n v = 
        let
        val w1 = LargeWord.<< ((Word8.toLargeWord (Word8Vector.sub (v,(n*4)))),0w24)
        val w2 = LargeWord.<< ((Word8.toLargeWord (Word8Vector.sub (v,(n*4)+1))),0w16)
        val w3 = LargeWord.<< ((Word8.toLargeWord (Word8Vector.sub (v,(n*4)+2))),0w8)
        val w4 = (Word8.toLargeWord (Word8Vector.sub (v,(n*4)+3)))
        val w12 = (LargeWord.orb (w1,w2))
        val w34 = (LargeWord.orb (w3,w4))
        val w = Word.fromLargeWord (LargeWord.orb (w12,w34))
        in w end
    fun valToWordList (XProtTypes.RAW_DATA{format=XProtTypes.Raw32,data}) =
        let
        fun rxt n a = if (n>=0) then (rxt (n-1) ((getWord32 n data)::a)) else a
        val wl = (rxt (((Word8Vector.length data) div 4)-1) [])
        in wl end
      | valToWordList (XProtTypes.RAW_DATA{format=XProtTypes.Raw16,data}) =
        let
        fun rxt n a = if (n>=0) then (rxt (n-1) ((getWord16 n data)::a)) else a
        val wl = (rxt (((Word8Vector.length data) div 2)-1) [])
        in wl end
      | valToWordList (_) = []
    fun valToXIdList (pval) = 
        List.map (fn w => (XProtTypes.XID w)) (valToWordList pval)
    fun valToXId (pval) = 
        case (valToXIdList pval) of [x] => x | _ => XProtTypes.XID 0w0 (* << FIXME *)
    fun valToIntList (pval) = 
        List.map Word.toInt (valToWordList pval)
    fun valToInt (pval) = 
        case (valToIntList pval) of [n] => n | _ => 0
    fun valToIntPair (pval) = 
        case (valToIntList pval) of [i,j] => (i,j) | _ => (0,0) (* << FIXME? *)
    fun valToAtomList (pval) =
        List.map (fn w => (XProtTypes.XAtom w)) (valToWordList pval)
    fun valToTgtList (dpy,pval) =
        List.map (fn a => (atomToTgt dpy a)) (valToAtomList pval)
    fun valToWord32 (pval) =
        case (valToWordList pval) of [w] => (Word32.fromLargeWord (Word.toLargeWord w)) | _ => 0w0
    
    fun isIncr dpy (SOME (XProtTypes.PROP_VAL{typ,value})) = 
            let
            val (XProtTypes.XAtom tw) = typ
            val (XProtTypes.XAtom iw) = (XAtoms.internAtom dpy "INCR")
            val _ = TextIO.print ("Type of property: "^(XAtoms.nameOfAtom dpy typ)^"\n")
            in (tw=iw) end
      | isIncr dpy (_) = false
    
    (* getIncr : display -> Property.property * Property.prop_val -> value *)
    fun getIncr (prop,SOME (XProtTypes.PROP_VAL{typ,value=XProtTypes.RAW_DATA{format,data}})) =
            let
            (* val lb = valToInt (value) *)
            val _ = TextIO.print "Getting INCR property ...\n"
            fun loop ((Property.NewValue,t),typ,XProtTypes.RAW_DATA{format,data}) =
                (TextIO.print "loop received new value\n";
                 case (Property.getProperty prop) of
                    (SOME (Property.PROP_VAL{typ,
                        value=XProtTypes.RAW_DATA{format=f',data=d'}})) =>
                        if ((Word8Vector.length d')>0) 
                        then (Property.getDeleteProperty prop;
                            loop ((CML.sync (Property.watchProperty prop)),
                                typ,XProtTypes.RAW_DATA{format=f',
                                    data=(Word8Vector.concat [d',data])}))
                        else (SOME (XProtTypes.PROP_VAL{typ=typ,
                            value=XProtTypes.RAW_DATA{format=format,data=data}}))
                  | _ => (SOME (XProtTypes.PROP_VAL{typ=typ,
                            value=XProtTypes.RAW_DATA{format=format,data=data}})))
              | loop (_,typ,pd) = loop ((CML.sync (Property.watchProperty prop)),typ,pd)
            in ((Property.getDeleteProperty prop); 
                 TextIO.print ("getIncr deleted property "^(Property.toString prop)^" initially\n");
                (loop ((CML.sync (Property.watchProperty prop)),
                    typ,XProtTypes.RAW_DATA{format=format,
                    data=(Word8Vector.fromList [])}))) end
       | getIncr (_,_) = NONE
       
    fun valHexDump (XProtTypes.RAW_DATA{format=XProtTypes.Raw8,data}) = 
            "Raw8: "^(Word8Vector.foldl (fn (w,s) => (s^" 0w"^(Word8.toString w))) "" data)
      | valHexDump (XProtTypes.RAW_DATA{format=XProtTypes.Raw16,data}) = 
            "Raw16: "^(Word8Vector.foldl (fn (w,s) => (s^" 0w"^(Word8.toString w))) "" data)
      | valHexDump (XProtTypes.RAW_DATA{format=XProtTypes.Raw32,data}) = 
            "Raw32: "^(Word8Vector.foldl (fn (w,s) => (s^" 0w"^(Word8.toString w))) "" data)
    
    fun propValToVal dpy (_,NONE) = (NONE)
      | propValToVal dpy (tgt,SOME (XProtTypes.PROP_VAL{typ,value})) = (
         (* TextIO.print ("propValToVal("^(XAtoms.nameOfAtom dpy (tgtToAtom dpy tgt))^","^
            "SOME(PROP_VAL{typ="^(XAtoms.nameOfAtom dpy typ)
            ^",value="^(valHexDump value)^"\n"); *)
        case ((XAtoms.nameOfAtom dpy (tgtToAtom dpy tgt)),(XAtoms.nameOfAtom dpy typ)) of
             ("TARGETS","ATOM")       => SOME (Val_TARGETS  (valToTgtList (dpy,value)))   
           | ("MULTIPLE",_)           => NONE (*(Val_MULTIPLE [])          fixme *)   
           | ("TIMESTAMP","INTEGER")  => SOME (Val_TIMESTAMP (XTime.XT(valToWord32 value)))         
           | ("LIST_LENGTH","INTEGER") => SOME (Val_LIST_LENGTH    (valToInt value))         
           | ("PIXMAP","DRAWABLE")    => SOME (Val_PIXMAP         (valToXIdList value))              
           | ("DRAWABLE","DRAWABLE")  => SOME (Val_DRAWABLE       (valToXIdList value))            
           | ("BITMAP","BITMAP")      => SOME (Val_BITMAP         (valToXIdList value))              
           | ("FOREGROUND","PIXEL")   => SOME (Val_FOREGROUND     (valToXId value))          
           | ("BACKGROUND","PIXEL")   => SOME (Val_BACKGROUND     (valToXId value))          
           | ("COLORMAP","COLORMAP")  => SOME (Val_COLORMAP       (valToXIdList value))            
           | ("ODIF","STRING")        => SOME (Val_ODIF           (valToString value))                
           | ("OWNER_OS","STRING")    => SOME (Val_OWNER_OS       (valToString value))            
           | ("FILE_NAME","STRING")   => SOME (Val_FILE_NAME      (valToString value))           
           | ("HOST_NAME","STRING")   => SOME (Val_HOST_NAME      (valToString value))           
           | ("CHARACTER_POSITION","SPAN") => SOME (Val_CHARACTER_POSITION (valToIntPair value))  
           | ("LINE_NUMBER","SPAN")   => SOME (Val_LINE_NUMBER    (valToIntPair value))         
           | ("COLUMN_NUMBER","SPAN") => SOME (Val_COLUMN_NUMBER  (valToIntPair value))       
           | ("LENGTH","INTEGER")     => SOME (Val_LENGTH         (valToInt value))              
           | ("USER","STRING")        => SOME (Val_USER           (valToString value))                
           | ("PROCEDURE","STRING")   => SOME (Val_PROCEDURE      (valToString value))           
           | ("MODULE","STRING")      => SOME (Val_MODULE         (valToString value))              
           | ("PROCESS","INTEGER")    => SOME (Val_PROCESS        (valToInt value))             
           | ("TASK","INTEGER")       => SOME (Val_TASK           (valToInt value))                
           | ("CLASS","STRING")       => SOME (Val_CLASS          (valToString value))               
           | ("NAME","STRING")        => SOME (Val_NAME           (valToString value))                
           | ("CLIENT_WINDOW","WINDOW") => SOME (Val_CLIENT_WINDOW  (valToXId value))       
           | ("DELETE",_)             => SOME (Val_DELETE)              
           | ("INSERT_SELECTION",_)   => SOME (Val_INSERT_SELECTION)    
           | ("INSERT_PROPERTY",_)    => SOME (Val_INSERT_PROPERTY) 
           | (_,"STRING")             => SOME (Val_STRING         (valToString value))             
           | (_,"COMPOUND_TEXT")      => SOME (Val_COMPOUND_TEXT  (valToString value))       
           | _ => NONE)
         
    type convertfn = target -> value option
    
    datatype sel_req = ReleaseEvt of ((unit CML.event) SyncVar.ivar)
                     | ReleaseCmd
                     
    datatype selection_handle = ISH of (sel_req CML.chan)
    
    fun releaseEvt (ISH ish) = 
        let
        val iv = SyncVar.iVar()
        val _  = CML.send (ish,ReleaseEvt iv)
        in (SyncVar.iGet iv) end
    fun releaseSelection (ISH ish) = (CML.send (ish,ReleaseCmd))
    
    fun acquireSelection (w,s,t,cvt:convertfn) =
        let
        val dpy = Window.displayOfWin w
        
        val p = Property.unusedProperty w
        fun f () = ((CML.sync (Property.watchProperty p));
                    TextIO.print "test property changed\n")
        val _ = CML.spawn f
        val _ = Property.getDeleteProperty p
        
        in
        case (Selection.acquireSelection (w,selToAtom dpy s,t)) of
            SOME sh => 
                let
                val ish = CML.channel()
                val srq = Selection.selectionReqEvt sh
                val srl = Selection.selectionRelEvt sh
                fun loop (rc) =
                    let
                    fun incrReply (repfn,prop,NONE) = (repfn NONE)
                      | incrReply (repfn,prop,SOME (pv as (XProtTypes.PROP_VAL{typ,
                            value=XProtTypes.RAW_DATA{format,data}}))) =
                        let
                        val (Display.DPY{xdpy=XDisplay.XDPY{max_req_len,...},...}) = dpy
                        (* reserve 8*4 bytes for header info; 5 should be enough...*)
                        val mxl = ((max_req_len-8)*4) (* max size of value *)
                        val mxl = (mxl div 1024) (* for testing *)
                        val _ = TextIO.print ("max_req_len="^
                            (Int.toString max_req_len)^",mxl="^(Int.toString mxl)^
                            ",length="^(Int.toString (Word8Vector.length data))^"\n")
                        fun repSub (i) = 
                            (TextIO.print ("repSub "^(Int.toString i)^",watching "^(Property.toString prop)^"\n");
                             case (CML.sync (Property.watchProperty prop)) of
                              (Property.Deleted,_) => 
                                (TextIO.print "incrReply notified of deleted property\n";
                                if (((Word8Vector.length data)-i)>mxl)
                                then (Property.setProperty (prop,
                                   (XProtTypes.PROP_VAL{typ=typ,
                                    value=XProtTypes.RAW_DATA{format=format,
                                    data=(Word8VectorSlice.vector
                                        (Word8VectorSlice.slice (data,i,SOME mxl)))}}));
                                    repSub (i+mxl))
                                else (repfn (SOME (XProtTypes.PROP_VAL{typ=typ,
                                    value=XProtTypes.RAW_DATA{format=format,
                                    data=(Word8VectorSlice.vector
                                        (Word8VectorSlice.slice (data,i,NONE)))}})))
                                )
                            | _ => (TextIO.print "incrReply notified of other property change\n";
                                    repSub(i)) )
                        in if ((Word8Vector.length data)>mxl)
                           then (TextIO.print ("Setting INCR on "^(Property.toString prop)^"\n");
                                 Property.setProperty (prop,
                                    (intListToPropVal (
                                        (XAtoms.internAtom dpy "INCR"),
                                        [Int.toLarge (Word8Vector.length data)])));
                                 CML.spawn (fn () => (
                                    repSub 0;
                                    Property.setProperty (prop,(XProtTypes.PROP_VAL{typ=typ,
                                        value=XProtTypes.RAW_DATA{format=format,
                                        data=(Word8Vector.fromList [])}}))
                                 )); ()
                                )
                           else repfn (SOME pv)
                        end
                    fun cvtAndReply (rqwin,[],_) = ()
                      | cvtAndReply (rqwin,_,[]) = ()
                      | cvtAndReply (rqwin,(Tgt_MULTIPLE _)::ts,(prop,reply)::rs) =
                            (* retrieve the target types and properties from the 
                             * property. Note that atomToTgt called below will not
                             * have populated the target types desired into the
                             * Tgt_MULTIPLE datatype. *)
                            let
                            val pvo = Property.getProperty prop
                            val al : XProtTypes.atom list = 
                                    (case (pvo) of 
                                        (NONE) => ([])
                                      | (SOME (XProtTypes.PROP_VAL{typ,value})) => 
                                                (valToAtomList (value)))
                            fun alToTPListPair ([]: XProtTypes.atom list) = ([],[])
                              | alToTPListPair (a::[]) = ([],[])
                              | alToTPListPair (ta::pa::al) = 
                                    let
                                    val (ts,rs) = alToTPListPair al
                                    val p = Property.property (rqwin,pa)
                                    fun r (SOME pv) = (Property.setProperty (p,pv))
                                      | r (NONE)    = ()
                                    in (((atomToTgt dpy ta)::ts,(p,r)::rs)) end
                            val (ts,rs) = alToTPListPair (al)
                            in (cvtAndReply (rqwin,ts,rs)); incrReply(reply,prop,pvo) end
                      | cvtAndReply (rqwin,(Tgt_TIMESTAMP)::ts,(prop,reply)::rs) =
                            let
                            val vt = (valToPropVal dpy (SOME (Val_TIMESTAMP t)))
                            in (reply vt; cvtAndReply (rqwin,ts,rs)) end
                      | cvtAndReply (rqwin,(Tgt_TARGETS)::ts,(prop,reply)::rs) =
                             let
                             val l = (case (cvt (Tgt_TARGETS)) of SOME (Val_TARGETS l) => l | _ => [])
                             val v = (SOME (Val_TARGETS (Tgt_TIMESTAMP::l)))
                             in (incrReply(reply,prop,(valToPropVal dpy v));
                                 cvtAndReply (rqwin,ts,rs)) end 
                      | cvtAndReply (rqwin,(tgt)::ts,(prop,reply)::rs) =
                             let
                             val vo: value option = (cvt tgt)
                             in (incrReply(reply,prop,(valToPropVal dpy vo));
                                 cvtAndReply (rqwin,ts,rs)) end
                    fun handleReq {target=ta,time,reply,win,property} =
                        let
                        val DrawTypes.WIN{id,scr,scr_depth,draw_cmd} = w
                        val rw = DrawTypes.WIN{id=win,scr=scr,scr_depth=scr_depth,draw_cmd=draw_cmd}
                        val prop = Property.property (rw,property)
                        in ((cvtAndReply (rw,[(atomToTgt dpy ta)],[(prop,reply)]); loop(rc))) end
                    fun handleRel (NONE) () = ()
                      | handleRel (SOME rc) () = CML.send (rc,())
                    fun handleHdl (ReleaseEvt iv) =
                        let
                        val rc = CML.channel()
                        val _ = SyncVar.iPut(iv,(CML.recvEvt rc))
                        in loop (SOME rc) end
                      | handleHdl (ReleaseCmd) =
                        (Selection.releaseSelection sh)
                    in CML.select [
                        CML.wrap(srq,(handleReq)),
                        CML.wrap(srl,(handleRel rc)),
                        CML.wrap(CML.recvEvt ish,(handleHdl))
                    ] end  
                (* FIXME: add support for MULTIPLE targets and incr. transfers. *)
                in XDebug.xspawn ("ICCCSelectionServer", (fn () => (loop NONE)) );
                  SOME (ISH ish) end
          | NONE => NONE
       end

    fun convertString str =
        let
        fun cvt (Tgt_STRING)  = SOME (Val_STRING str)
          | cvt (Tgt_TARGETS) = SOME (Val_TARGETS [Tgt_STRING,
                                                   Tgt_TARGETS,
                                                   Tgt_HOST_NAME,
                                                   Tgt_LENGTH])
          | cvt (Tgt_HOST_NAME) = SOME (Val_HOST_NAME (NetHostDB.getHostName ()))
          | cvt (Tgt_LENGTH)  = SOME (Val_LENGTH (String.size str))
          | cvt (_)           = NONE
        in cvt end
      
    fun requestSelection (w,s,tgt,ti) =
        let
        val dpy = Window.displayOfWin w
        val s = selToAtom dpy s
        val ta = tgtToAtom dpy tgt
        fun mkProp (tgt as Tgt_MULTIPLE tgl) =
                let
                val aa = XAtoms.internAtom dpy "ATOM"
                val p = Property.unusedProperty w
                val ta = tgtToAtom dpy tgt
                fun cvt ([],al) = al
                  | cvt (tgt::ts,al) = 
                    let
                    val ta = (tgtToAtom dpy tgt)
                    val pa = (Property.nameOfProp (mkProp tgt))
                    in (cvt (ts,[ta,pa]@al)) end
                val pv = (atomListToPropVal (aa, (cvt (tgl,[]))))
                val XProtTypes.PROP_VAL{typ,value} = pv
                val _ = Property.setProperty (p,pv)
                in p end 
          | mkProp (_) = 
                let
                val p = Property.unusedProperty w
                val _ = TextIO.print ("Allocating property "^(Property.toString p)^"\n")
                in p end
        val p = mkProp tgt
        fun list2pairlist (a::b::cs) = (a,b)::(list2pairlist cs)
          | list2pairlist (_) = []
        fun getProp (Tgt_MULTIPLE _,p,SOME (Property.PROP_VAL{typ,value})) =
              SOME (Val_MULTIPLE (List.mapPartial 
                (fn (ta,pa) => getProp (atomToTgt dpy ta,(Property.property (w,pa)),
                    Property.getProperty (Property.property (w,pa)) ) ) 
                (list2pairlist (valToAtomList value)) ))
          | getProp (Tgt_MULTIPLE _,p,NONE) = NONE
          | getProp (tgt,p,pvo) = 
              (propValToVal dpy (tgt,
                (if (isIncr dpy pvo) 
                 then (getIncr (p,pvo)) 
                 else pvo)))
        fun rcv pvo = 
            let
            val v = getProp (tgt,p,pvo)
            val _ = Property.deleteProperty p
            in v end
        in (CML.wrap((Selection.requestSelection{win=w,selection=s,
             target=ta,property=(Property.nameOfProp p),time=ti}),rcv)) end
                 
    fun requestSelectionString (w,s,ti) =
        CML.wrap ((requestSelection(w,s,Tgt_STRING,ti)),
            fn so => (case so of (SOME (Val_STRING(s))) => SOME s | _ => NONE) )
        
    fun cutBuffer0 win = Property.property (win,StdAtoms.atom_CUT_BUFFER0)
    fun cutBuffer1 win = Property.property (win,StdAtoms.atom_CUT_BUFFER1)
    fun cutBuffer2 win = Property.property (win,StdAtoms.atom_CUT_BUFFER2)
    fun cutBuffer3 win = Property.property (win,StdAtoms.atom_CUT_BUFFER3)
    fun cutBuffer4 win = Property.property (win,StdAtoms.atom_CUT_BUFFER4)
    fun cutBuffer5 win = Property.property (win,StdAtoms.atom_CUT_BUFFER5)
    fun cutBuffer6 win = Property.property (win,StdAtoms.atom_CUT_BUFFER6)
    fun cutBuffer7 win = Property.property (win,StdAtoms.atom_CUT_BUFFER7)
    
    fun storeCutBuffer(win,str) = 
        Property.setProperty(cutBuffer0 win, 
            Property.PROP_VAL{typ=StdAtoms.atom_STRING,
                value=XProtTypes.RAW_DATA{format=XProtTypes.Raw8,data=(Byte.stringToBytes str)}})
        
    fun fetchCutBuffer(win) = 
        case (Property.getProperty (cutBuffer0 win)) of 
            SOME (Property.PROP_VAL{typ,value=XProtTypes.RAW_DATA{
                    format=XProtTypes.Raw8,data}}) =>
                        if (typ=(StdAtoms.atom_STRING))
                        then SOME (Byte.bytesToString data)
                        else NONE
          | _ => NONE
    
    fun rotateCutBuffer(win,n) =
        Property.rotateProperties([cutBuffer0 win,
                                   cutBuffer1 win,
                                   cutBuffer2 win,
                                   cutBuffer3 win,
                                   cutBuffer4 win,
                                   cutBuffer5 win,
                                   cutBuffer6 win,
                                   cutBuffer7 win],n)
  end (* Selection *)
